#include <mail/ratesrv/src/storage/repository.h>

#include <gtest/gtest.h>

#include <memory>
#include <string>

namespace {

using namespace testing;
using namespace NRateSrv;
using namespace NRateSrv::NStorage;

class TTestRepository : public Test {
protected:
    const size_t DefaultPartCount = 10;

protected:
    void SetUp() override {
        Repository = std::make_unique<TRepository>(DefaultPartCount);
    }

    void AddUpdateHandle(std::string key, TRepository::TUpdateCallback callback) {
        TRepository::TUpdateHandle handle{std::move(key), std::move(callback)};
        Handles.push_back(std::move(handle));
    }

    void Update(bool createNonExistent) {
        TRepository::TUpdateHandles handles;
        std::swap(handles, Handles);
        Repository->Update(std::move(handles), createNonExistent);
    }

    TCounter MakeDefaultCounter() {
        TCounter counter{10, TClock::now(), 11};
        return counter;
    }

protected:
    std::unique_ptr<TRepository> Repository;
    TRepository::TUpdateHandles Handles;
};

TEST_F(TTestRepository, GetNonExistent) {
    AddUpdateHandle("test1", [](const TCounter* counter) -> TCounter {
        EXPECT_EQ(counter, nullptr);
        return {};
    });
    Update(false);

    // We need to make sure the repository is empty.
    AddUpdateHandle("test1", [](const TCounter* counter) -> TCounter {
        EXPECT_EQ(counter, nullptr);
        return {};
    });
    Update(false);
}

TEST_F(TTestRepository, CreateAndGet) {
    std::unordered_map<std::string, TCounter> counters;
    auto now = TClock::now();

    counters.emplace("test1", TCounter{10, now, 100});
    counters.emplace("test2", TCounter{20, now, 200});
    counters.emplace("test3", TCounter{30, now, 300});

    for (const auto& [key, counter] : counters) {
        AddUpdateHandle(key, [counter = counter](const TCounter* origCounter) -> TCounter {
            EXPECT_EQ(origCounter, nullptr);
            return counter;
        });
    }
    Update(true);
    EXPECT_EQ(Repository->Size(), 3u);

    for (const auto& [key, counter] : counters) {
        AddUpdateHandle(key, [counter = counter](const TCounter* origCounter) -> TCounter {
            EXPECT_NE(origCounter, nullptr);
            EXPECT_TRUE(*origCounter == counter);
            return counter;
        });
    }
    Update(false);
}

TEST_F(TTestRepository, ReadEmpty) {
    EXPECT_EQ(Repository->Size(), 0u);
    Repository->Read([](const std::string&, const TCounter&) -> bool {
        EXPECT_TRUE(false);
        return true;
    });
}

TEST_F(TTestRepository, Read) {
    std::unordered_map<std::string, TCounter> counters;
    auto now = TClock::now();
    ASSERT_EQ(Repository->Size(), 0u);

    counters.emplace("test1", TCounter{10, now, 100});
    counters.emplace("test2", TCounter{20, now, 200});
    counters.emplace("test3", TCounter{30, now, 300});

    for (const auto& [key, counter] : counters) {
        AddUpdateHandle(key, [counter = counter](const TCounter*) -> TCounter {
            return counter;
        });
    }
    Update(true);

    Repository->Read([&counters](const std::string& key, const TCounter& counter) -> bool {
        auto it = counters.find(key);
        EXPECT_FALSE(it == counters.end());

        auto& origCounter = it->second;
        EXPECT_TRUE(origCounter == counter);

        counters.erase(it);

        return true;
    });

    EXPECT_EQ(counters.size(), 0u);
}

TEST_F(TTestRepository, ReadAndStop) {
    std::unordered_map<std::string, TCounter> counters;
    auto now = TClock::now();

    counters.emplace("test1", TCounter{10, now, 100});
    counters.emplace("test2", TCounter{20, now, 200});
    counters.emplace("test3", TCounter{30, now, 300});

    for (const auto& [key, counter] : counters) {
        AddUpdateHandle(key, [counter = counter](const TCounter*) -> TCounter {
            return counter;
        });
    }
    Update(true);

    bool stopped = false;
    Repository->Read([&stopped](const std::string&, const TCounter&) -> bool {
        EXPECT_FALSE(stopped);
        stopped = true;
        return false;
    });
}

TEST_F(TTestRepository, Erase) {
    std::unordered_map<std::string, TCounter> counters;
    auto now = TClock::now();

    counters.emplace("test1", TCounter{10, now, 100});
    counters.emplace("test2", TCounter{20, now, 200});
    counters.emplace("test3", TCounter{30, now, 300});

    for (const auto& [key, counter] : counters) {
        AddUpdateHandle(key, [counter = counter](const TCounter*) -> TCounter {
            return counter;
        });
    }
    Update(true);
    EXPECT_EQ(Repository->Size(), 3u);

    Repository->Erase({"test1", "test2"}, [&counters](const TCounter& counter) {
        auto counter1It = counters.find("test1");
        auto counter2It = counters.find("test2");
        if (counter1It != counters.end() && counter1It->second == counter) {
            counters.erase(counter1It);
            return true;
        } else if (counter2It != counters.end() && counter2It->second == counter) {
            return false;
        }
        EXPECT_TRUE(false);
        return false;
    });
    EXPECT_EQ(Repository->Size(), 2u);

    Repository->Read([&counters](const std::string& key, const TCounter& counter) -> bool {
        auto it = counters.find(key);
        EXPECT_FALSE(it == counters.end());

        auto& origCounter = it->second;
        EXPECT_TRUE(origCounter == counter);

        counters.erase(it);

        return true;
    });

    ASSERT_EQ(counters.size(), 0u);
}

}
