package ru.yandex.reminders.api.reminder;

import org.joda.time.DateTime;
import org.joda.time.DateTimeUtils;
import org.joda.time.DateTimeZone;
import org.junit.After;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.mock.web.MockHttpServletRequest;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.test.context.support.AnnotationConfigContextLoader;
import ru.yandex.bolts.collection.*;
import ru.yandex.bolts.function.Function2;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.http.ActionInvocationServlet;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.parse.BenderJsonNode;
import ru.yandex.misc.bender.parse.BenderParserUtils;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.net.uri.Uri2;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.misc.time.MoscowTime;
import ru.yandex.misc.web.servlet.mock.MockHttpServletResponse;
import ru.yandex.reminders.api.ApiBender;
import ru.yandex.reminders.api.ApiTestA3ContextConfiguration;
import ru.yandex.reminders.logic.event.*;
import ru.yandex.reminders.logic.reminder.Reminder;
import ru.yandex.reminders.logic.sup.SupClientSettings;
import ru.yandex.reminders.logic.sup.SupClientSettingsRegistry;
import ru.yandex.reminders.logic.sup.SupClientTestContextConfiguration;
import ru.yandex.reminders.logic.sup.SupPushRequest;
import ru.yandex.reminders.util.TestUtils;
import ru.yandex.reminders.util.UrlUtils;

@ContextConfiguration(
        loader = AnnotationConfigContextLoader.class,
        classes = {
                EventManagerTestContextConfiguration.class,
                ApiTestA3ContextConfiguration.class,
                ApiReminderContextConfiguration.class,
                SupClientTestContextConfiguration.class,
        }
)
@Ignore("SUBBOTNIK-1476")
@RunWith(SpringJUnit4ClassRunner.class)
public class ReminderActionsTest extends TestUtils {
    @Autowired
    private ActionInvocationServlet servlet;
    @Autowired
    private EventMdao eventMdao;
    @Autowired
    private SupClientSettingsRegistry registry;


    private final PassportUid uid = PassportUid.cons(1371858);
    private final String cid = "yandex-calendar";

    private final DateTime now = MoscowTime.dateTime(2018, 3, 9, 17, 10);

    @Before
    public void setup() {
        eventMdao.deleteEvents(EventsFilter.any().toMongoQuery(uid, cid));
        DateTimeUtils.setCurrentMillisProvider(now::getMillis);
    }

    @After
    public void teardown() {
        DateTimeUtils.setCurrentMillisProvider(System::currentTimeMillis);
    }

    @Test
    public void iterateWithoutInterval() {
        Cf.list("1", "2", "3", "4").forEach(id -> createEvent(id, Option.empty()));

        assertIterating(Cf.list(Cf.list("1", "2"), Cf.list("3", "4")), "count=2");
        assertIterating(Cf.list(Cf.list("4", "3", "2"), Cf.list("1")), "count=3&new-first=true");
        assertIterating(Cf.<ListF<String>>list(Cf.list("1", "2", "3", "4")), "count=5&new-first=false");
    }

    @Test
    public void iterateWithInterval() {
        Tuple2List<String, DateTime> datas = Tuple2List.fromPairs(
                "1", now.plusSeconds(15), "2", now.plusSeconds(10), "3", now.plusSeconds(25), "4", now);

        datas.forEach((id, ts) -> createEvent(id, Option.of(ts)));

        Function2<Integer, Integer, String> intervalF = (from, to) -> "interval="
                + UrlUtils.urlEncode(new InstantInterval(now.plusSeconds(from), now.plusSeconds(to)).toString());

        assertIterating(Cf.list(Cf.list("1"), Cf.list("2"), Cf.list("3")), "count=1&" + intervalF.apply(5, 25));
        assertIterating(Cf.list(Cf.list("3", "2"), Cf.list("1")), "count=2&new-first=true&" + intervalF.apply(10, 50));
        assertIterating(Cf.<ListF<String>>list(Cf.list("1", "3")), "count=3&new-first=false&" + intervalF.apply(15, 50));
    }

    @Test
    public void iterateWithSoonest() {
        Tuple2List<String, DateTime> datas = Tuple2List.fromPairs(
                "1", now.plusSeconds(15), "2", now.plusSeconds(10), "3", now.plusSeconds(10), "4", now);

        datas.forEach((id, ts) -> createEvent(id, Option.of(ts)));

        assertIterating(Cf.<ListF<String>>list(Cf.list("4", "2", "3", "1")), "count=10&soonest-first=true");
        assertIterating(Cf.list(Cf.list("4", "2"), Cf.list("3", "1")), "count=2&soonest-first=true");
        assertIterating(Cf.list(Cf.list("1", "3", "2"), Cf.list("4")), "count=3&soonest-first=false");
    }

    @Test
    public void autoXiva() {
        Mockito.when(registry.getO(Mockito.eq(cid))).thenReturn(
                Option.of(new SupClientSettings("sup", "sup", "sup", "PERSONAL", new SupPushRequest(), Option.of("xiva-token"))));

        ReminderData data = new ReminderData(
                Option.of("Name"), Option.empty(), Option.of(now),
                new ChannelsData(Option.empty(), Option.empty(), Option.empty(), Option.empty(),
                        Option.of(new ChannelsData.Sup(Option.empty(), Option.empty(), Option.of("uri"),
                                new ChannelsData.Sup.SupExtraFields()))),
                Option.empty());

        String id = request(HttpMethod.POST, ReminderIdInfo.class, "", Option.of(data)).getId();

        Option<Reminder> reminder = eventMdao.findEvent(new EventId(uid, cid, id))
                .flatMapO(e -> e.getReminders().find(Reminder::isXiva));

        Assert.some(reminder);
        Assert.some("uri", reminder.get().getUrl());
        Assert.isTrue(now.isEqual(reminder.get().getSendDate()));

        data = new ReminderData(Option.of("Renamed"), Option.empty(), Option.of(now),
                data.getChannels(), Option.empty());

        request(HttpMethod.PUT, ReminderIdInfo.class, "id=" + id, Option.of(data));

        Option<Event> event = eventMdao.findEvent(new EventId(uid, cid, id));

        Assert.some("Renamed", event.flatMapO(Event::getName));
        Assert.some(event.flatMapO(e -> e.getReminders().find(Reminder::isXiva)));
    }


    private void createEvent(String id, Option<DateTime> sendTs) {
        ReminderData data = new ReminderData(
                Option.of("Name of " + id), Option.of("Description of " + id),
                Option.of(sendTs.getOrElse(now).toDateTime(DateTimeZone.UTC)),
                new ChannelsData(Option.empty(), Option.empty(),
                        Option.of(new ChannelsData.Panel(Option.empty(), Option.empty(), Option.empty())),
                        Option.empty(), Option.empty()), Option.empty());

        request(HttpMethod.PUT, ReminderIdInfo.class, "id=" + UrlUtils.urlEncode(id), Option.of(data));
    }

    private void assertIterating(ListF<ListF<String>> expectedIds, String parameters) {
        IteratorF<ListF<String>> expectedIterator = expectedIds.iterator();
        Option<IterationKey> iterationKey = Option.empty();

        while (expectedIterator.hasNext()) {
            RemindersInfo response = getReminders(
                    parameters + iterationKey.map(k -> "&iteration-key=" + k.serialize()).mkString(""));

            Assert.equals(expectedIterator.next(), response.getIds());
            Assert.some(expectedIds.foldLeft(0L, (acc, lst) -> acc + lst.size()), response.getTotalCount());

            iterationKey = response.getIterationKey();
        }
        Assert.none(iterationKey, "End of items expected but got " + iterationKey.mkString(""));
        Assert.isFalse(expectedIterator.hasNext(), "More items expected");
    }

    private RemindersInfo getReminders(String parameters) {
        return request(HttpMethod.GET, RemindersInfo.class, parameters, Option.empty());
    }

    private <T> T request(HttpMethod method, Class<T> clazz, String parameters, Option<ReminderData> data) {
        Uri2 uri = Uri2.parse("/v1/" + uid + "/reminders/" + cid + "?" + parameters);

        MockHttpServletRequest request = new MockHttpServletRequest(
                method.name(), uri.onlyPathQueryFragment().toString());

        request.setPathInfo(uri.getPath());
        request.setParameters(uri.getQueryArgs().toMap());

        if (data.isPresent()) {
            request.setContent(ApiBender.mapper.serializeJson(data.get()));
        }
        request.addHeader("Accept", "application/json");

        MockHttpServletResponse response = new MockHttpServletResponse();

        try {
            servlet.service(request, response);

            if (!HttpStatus.is2xx(response.getStatus())) {
                throw new HttpException(response.getStatus(), response.getContentAsString());
            }
            BenderJsonNode rootNode = BenderParserUtils.json(response.getContentAsString());
            return ApiBender.mapper.parseJson(clazz, rootNode.getField("result").getOrThrow("No result"));

        } catch (Exception e) {
            throw ExceptionUtils.translate(e);
        }
    }
}
