package ru.yandex.reminders.logic.event;

import javax.annotation.Resource;

import org.bson.types.ObjectId;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.test.context.support.AnnotationConfigContextLoader;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.commune.bazinga.impl.JobStatus;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.reminders.api.reminder.Source;
import ru.yandex.reminders.logic.flight.FlightEventMeta;
import ru.yandex.reminders.logic.flight.FlightSource;
import ru.yandex.reminders.logic.reminder.Channel;
import ru.yandex.reminders.logic.reminder.Origin;
import ru.yandex.reminders.logic.reminder.PhoneNumber;
import ru.yandex.reminders.logic.reminder.Reminder;
import ru.yandex.reminders.logic.update.ActionInfo;
import ru.yandex.reminders.mongodb.BazingaOnetimeJobMdao;
import ru.yandex.reminders.util.TestUtils;

import static ru.yandex.reminders.logic.event.EventManagerTestCommon.FLIGHT_NUM;
import static ru.yandex.reminders.logic.event.EventManagerTestCommon.GEO_ID;
import static ru.yandex.reminders.logic.event.EventManagerTestCommon.PLANNED_TS;
import static ru.yandex.reminders.logic.event.EventManagerTestCommon.createEventJsonData;
import static ru.yandex.reminders.logic.event.EventManagerTestCommon.createFlightEventMeta;

@ContextConfiguration(
        loader = AnnotationConfigContextLoader.class,
        classes = EventManagerTestContextConfiguration.class
)
@Ignore("SUBBOTNIK-1476")
@RunWith(SpringJUnit4ClassRunner.class)
public class EventManagerTest extends TestUtils {
    @Resource
    private EventManager eventManager;
    @Resource
    private EventMdao eventMdao;
    @Resource
    private BazingaOnetimeJobMdao bazingaOnetimeJobMdao;

    @Before
    public void before() {
        eventMdao.dropAndCreateCollection();
    }

    @Test
    public void saveAndUpdate() {
        Reminder sms = createSmsReminder();
        Reminder email = createEmailReminder();

        Option<String> extId = Option.none();

        Option<String> name = Option.some("Event...");
        Option<String> description = Option.none();
        JsonObject dataJson = createEventJsonData();

        PassportUid uid = PassportUid.cons(1371858);
        String clientId = "EventManagerTest";

        FlightEventMeta meta = createFlightEventMeta();
        EventId id = new EventId(uid, clientId, extId);
        EventData data = new EventData(Option.some(Source.INTERNAL),
                name, description, Option.some(dataJson), Cf.list(sms, email));
        String externalId = eventManager.createOrUpdateEvent(id, data, actionInfo()).getExternalId();

        Event event = eventManager.findEvents(uid, clientId, EventsFilter.byExternalId(externalId)).single();
        Assert.equals(dataJson.serialize(), event.getData().get().serialize());
        Assert.equals(name, event.getName());
        Assert.hasSize(2, event.getReminders());

        Assert.hasSize(0,
                eventMdao.findFlightEvents(Cf.<PassportUid>list(), FLIGHT_NUM, GEO_ID, PLANNED_TS, FlightSource.RASP));

        data = new EventData(Option.some(Source.INTERNAL),
                Option.<String>none(), description, Option.some(dataJson), Cf.list(sms), Option.some(meta));
        eventManager.updateEvent(event, data, actionInfo());

        event = eventManager.findEvents(uid, clientId, EventsFilter.byExternalId(externalId)).single();
        Assert.none(event.getName());
        Assert.hasSize(1, event.getReminders());

        // findFlightEvents() tests
        Assert.hasSize(1,
                eventMdao.findFlightEvents(Cf.<PassportUid>list(), FLIGHT_NUM, GEO_ID, PLANNED_TS, FlightSource.RASP));
        Assert.hasSize(0,
                eventMdao.findFlightEvents(Cf.list(uid), FLIGHT_NUM, GEO_ID, PLANNED_TS, FlightSource.RASP));
        Assert.hasSize(1,
                eventMdao.findFlightEvents(Cf.list(PassportUid.cons(9999)), FLIGHT_NUM, GEO_ID, PLANNED_TS, FlightSource.RASP));

        Assert.hasSize(0,
                eventMdao.findFlightEvents(Cf.<PassportUid>list(), FLIGHT_NUM, GEO_ID, PLANNED_TS, FlightSource.IEX));
        Assert.hasSize(0,
                eventMdao.findFlightEvents(Cf.<PassportUid>list(), FLIGHT_NUM, GEO_ID,
                        PLANNED_TS.plus(Duration.standardMinutes(1)), FlightSource.RASP));
        Assert.hasSize(0,
                eventMdao.findFlightEvents(Cf.<PassportUid>list(), FLIGHT_NUM, GEO_ID + 1, PLANNED_TS, FlightSource.RASP));
        Assert.hasSize(0,
                eventMdao.findFlightEvents(Cf.<PassportUid>list(), FLIGHT_NUM + "_a", GEO_ID, PLANNED_TS, FlightSource.RASP));
    }

    @Test
    public void reschedule() {
        EventId eventId = eventManager.createOrUpdateEvent(
                new EventId(PassportUid.cons(1371858), "EventManagerTest", Option.<String>none()),
                new EventData(
                        Option.some(Source.INTERNAL),
                        Option.some("Event..."), Option.<String>none(), Option.<JsonObject>none(),
                        Cf.list(createSmsReminder())), actionInfo()).getId();
        Event event = eventManager.findEvent(eventId).get();
        ListF<Reminder> reminders = event.getReminders();
        Assert.sizeIs(1, reminders);

        ObjectId reminderId = reminders.first().getId();

        ListF<OnetimeJob> job = bazingaOnetimeJobMdao
                .findByActiveUniqueId(BazingaOnetimeJobMdao.reminderIdToActiveUniqueId(reminderId));
        Assert.sizeIs(1, job);
        Assert.equals(JobStatus.READY, job.first().getValue().getStatus());

        Instant newScheduleTime = Instant.now().plus(Duration.standardDays(1));
        Assert.isTrue(bazingaOnetimeJobMdao.rescheduleReminderSendTaskByReminderIdAndStatus(
                reminderId, JobStatus.READY, newScheduleTime));

        findAndCheckJob(reminderId, newScheduleTime);

        // check "job not found" cases
        Assert.isFalse(bazingaOnetimeJobMdao.rescheduleReminderSendTaskByReminderIdAndStatus(
                ObjectId.get(), JobStatus.READY, newScheduleTime));
        Assert.isFalse(bazingaOnetimeJobMdao.rescheduleReminderSendTaskByReminderIdAndStatus(
                reminderId, JobStatus.COMPLETED, newScheduleTime));

        findAndCheckJob(reminderId, newScheduleTime);
    }

    private void findAndCheckJob(ObjectId reminderId, Instant expectedScheduleTime) {
        ListF<OnetimeJob> jobs;
        jobs = bazingaOnetimeJobMdao.findByActiveUniqueId(BazingaOnetimeJobMdao.reminderIdToActiveUniqueId(reminderId));
        Assert.sizeIs(1, jobs);
        OnetimeJob job = jobs.first();
        Assert.equals(JobStatus.READY, job.getValue().getStatus());
        Assert.equals(expectedScheduleTime, job.getScheduleTime());
    }

    @Test
    public void deleteReminders() {
        final Reminder sms = createSmsReminder();
        final Reminder email = createEmailReminder();

        final PassportUid uid = PassportUid.cons(1371858);
        final String clientId = "DeleteRemindersTest";
        final String extId = "ext1";
        EventsFilter filter = EventsFilter.byExternalId(extId);
        ListF<Long> idxs = Cf.list(11L, 12L);


        idxs.forEach(new Function1V<Long>() {
            public void apply(Long idx) {
                EventId id = new EventId(uid, clientId, extId, idx);
                EventData data = new EventData(
                        Option.some(Source.INTERNAL),
                        Option.<String>none(), Option.<String>none(), Option.<JsonObject>none(),
                        Cf.list(sms, email));
                eventManager.createOrUpdateEvent(id, data, actionInfo());
            }
        });

        ListF<Event> events = eventManager.findEvents(uid, clientId, filter);
        Assert.sizeIs(2, events);
        Assert.isTrue(events.forAll(Event.getRemindersF().andThen(Cf.List.sizeF()).andThenEquals(2)));

        eventManager.deleteReminders(uid, clientId, filter, Channel.SMS);

        events = eventManager.findEvents(uid, clientId, filter);
        Assert.sizeIs(2, events);
        Assert.isTrue(events.forAll(Event.getRemindersF().andThen(Cf.List.sizeF()).andThenEquals(1)));
        Assert.isTrue(events.forAll(Event.getRemindersF().andThen(Cf.List.<Reminder>firstF())
                .andThen(Reminder.channelIsF(Channel.EMAIL))));
    }

    @Test
    public void eventFilter() {
        InstantInterval interval = new InstantInterval(DateTime.now().minusDays(1), DateTime.now().plusDays(1));
        ListF<Event> events = Cf.list(
                saveReminders(createSmsReminderWithSendDate(DateTime.now().minusDays(3))),
                saveReminders(createSmsReminderWithSendDate(DateTime.now())),
                saveReminders(createSmsReminderWithSendDate(DateTime.now().plusDays(3))));

        EventsFilter filter = EventsFilter.any().andByReminderSendTs(interval);
        events = eventManager.findEvents(events.first().getUid(), events.first().getClientId(), filter);

        Assert.hasSize(1, events);
    }



    private Event saveReminders(Reminder... reminders) {
        PassportUid uid = PassportUid.cons(1371858);
        String clientId = "EventManagerTest";

        EventId id = new EventId(uid, clientId);
        EventData data = new EventData(
                Option.some(Source.INTERNAL),
                Option.<String>none(), Option.<String>none(), Option.<JsonObject>none(), Cf.x(reminders));

        return eventManager.createOrUpdateEvent(id, data, actionInfo());
    }

    private Reminder createSmsReminder() {
        return createSmsReminderWithSendDate(DateTime.now().plusHours(2));
    }

    private Reminder createSmsReminderWithSendDate(DateTime sendDate) {
        Option<Integer> offset = Option.some(Random2.R.nextInt());
        Option<PhoneNumber> phone = Option.some(PhoneNumber.number(Random2.R.nextLong()));
        Option<String> text = Option.some(Random2.R.nextLetters(127));

        return Reminder.sms(sendDate, offset, Option.<Origin>none(), phone, text);
    }

    private Reminder createEmailReminder() {
        DateTime sendDate = DateTime.now();
        Option<Integer> offset = Option.some(Random2.R.nextInt());
        Option<String> from = Option.some("John Doe");
        Option<Email> email = Option.some(new Email(Random2.R.nextLetters(15) + "@yandex.ru"));
        Option<String> subject = Option.some(Random2.R.nextLetters(35));
        Option<String> bodyHtml = Option.some(Random2.R.nextLetters(512));
        Option<String> bodyText = Option.some(Random2.R.nextLetters(512));

        return Reminder.email(sendDate, offset, Option.some(Origin.USER), from, email, subject, bodyText, bodyHtml);
    }

    private static ActionInfo actionInfo() {
        return new ActionInfo(Instant.now(), Random2.R.nextLetters(12), Option.<String>none());
    }
}
