package ru.yandex.reminders.logic.flight;

import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.mockito.Mockito;
import org.mockito.stubbing.Answer;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.bazinga.impl.OnetimeUtils;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.time.MoscowTime;
import ru.yandex.reminders.api.flight.FlightDataConverter;
import ru.yandex.reminders.api.flight.MailFlightData;
import ru.yandex.reminders.api.flight.MailReminderData;
import ru.yandex.reminders.api.unistat.UnistatConfiguration;
import ru.yandex.reminders.boot.InitContextConfiguration;
import ru.yandex.reminders.logic.event.*;
import ru.yandex.reminders.logic.flight.airport.AirportManager;
import ru.yandex.reminders.logic.reminder.Channel;
import ru.yandex.reminders.logic.reminder.Reminder;
import ru.yandex.reminders.logic.update.ActionInfo;
import ru.yandex.reminders.logic.user.UserContextConfiguration;
import ru.yandex.reminders.mongodb.BazingaOnetimeJobMdao;
import ru.yandex.reminders.mongodb.TestMdaoContextConfiguration;
import ru.yandex.reminders.tvm.TvmClientTestConfiguration;
import ru.yandex.reminders.util.TestUtils;
import ru.yandex.reminders.worker.DeleteCloudApiFlightsTask;
import ru.yandex.reminders.worker.DeleteCloudApiFlightsTaskParameters;

@ContextConfiguration(classes = {
        InitContextConfiguration.class,
        TvmClientTestConfiguration.class,
        TestMdaoContextConfiguration.class,
        UserContextConfiguration.class,
        EventContextConfiguration.class,
        FlightReminderManagerContextConfiguration.class,
        UnistatConfiguration.class,
})
@Ignore("SUBBOTNIK-1476")
public class FlightReminderManagerTest extends TestUtils {
    @Autowired
    private FlightReminderManager flightReminderManager;
    @Autowired
    private BazingaOnetimeJobMdao bazingaOnetimeJobMdao;
    @Autowired
    private EventMdao eventMdao;

    @Before
    public void cleanUp() {
        eventMdao.dropAndCreateCollection();
        bazingaOnetimeJobMdao.removeAll();
    }

    @Test
    public void scheduleCloudApiReminderAndDeletion() {
        AirportManager airportManager = Mockito.mock(AirportManager.class, (Answer) invocation -> Option.none());

        DateTime now = DateTime.now(MoscowTime.TZ);

        MailFlightData.Builder builder = MailFlightData.builder()
                .setFlightNumber("SU2030").setDepartureCity("Москва").setArrivalCity("Хошимин")
                .setDepartureTz(MoscowTime.TZ)
                .setPlannedArrivalDateTime(now);

        ListF<MailFlightData> flightDatas = Cf.list(
                builder.setDepartureDateTime(now.minusMonths(2).toLocalDateTime()).build(),
                builder.setDepartureDateTime(now.plusHours(1).toLocalDateTime()).build(),
                builder.setDepartureDateTime(now.plusMonths(2).toLocalDateTime()).build());

        ListF<FlightEventMeta> flights = flightDatas.map(
                flight -> FlightDataConverter.toFlightEventMeta("mid", airportManager, flight));

        MailReminderData remindersData = MailReminderData.empty();
        PassportUid uid = PassportUid.cons(44902531L);
        String externalId = "ticket-777";
        ActionInfo actionInfo = new ActionInfo(now.toInstant(), "", Option.<String>none());

        flightReminderManager.createOrUpdateFlightReminders(uid, externalId, remindersData, flights, actionInfo);

        ListF<Event> events = eventMdao.findEvents(
                EventsFilter.byExternalId(externalId).toMongoQuery(uid, SpecialClientIds.FLIGHT));

        Assert.hasSize(3, events);

        events = events.sortedBy(Event::getIdx);

        Function<Event, ListF<Reminder>> cloudApiRemindersF =
                e -> e.getReminders().filter(Reminder.channelIsF(Channel.CLOUD_API));

        Assert.isEmpty(cloudApiRemindersF.apply(events.get(0)));
        Assert.hasSize(1, cloudApiRemindersF.apply(events.get(1)));
        Assert.hasSize(1, cloudApiRemindersF.apply(events.get(2)));

        Reminder reminder = cloudApiRemindersF.apply(events.get(1)).single();
        Assert.equals(now, reminder.getSendDate());
        Assert.equals(Channel.CLOUD_API, reminder.getChannel());

        Assert.hasSize(1, bazingaOnetimeJobMdao.findByActiveUniqueId(
                BazingaOnetimeJobMdao.reminderIdToActiveUniqueId(reminder.getId())));

        reminder = cloudApiRemindersF.apply(events.get(2)).single();
        Assert.isTrue(reminder.getSendDate().isAfter(now));
        Assert.equals(Channel.CLOUD_API, reminder.getChannel());

        Assert.hasSize(1, bazingaOnetimeJobMdao.findByActiveUniqueId(
                BazingaOnetimeJobMdao.reminderIdToActiveUniqueId(reminder.getId())));

        flightReminderManager.createOrUpdateFlightReminders(uid, externalId, remindersData, flights.rtake(1), actionInfo);

        ListF<OnetimeJob> jobs = bazingaOnetimeJobMdao.findByTaskId(DeleteCloudApiFlightsTask.TASK_ID.getId());
        Assert.hasSize(1, jobs);

        DeleteCloudApiFlightsTaskParameters params = (DeleteCloudApiFlightsTaskParameters) OnetimeUtils.parseParameters(
                new DeleteCloudApiFlightsTask(), jobs.single().getParameters());

        Assert.equals(Cf.list(events.get(0).getIdx(), events.get(1).getIdx()), params.getIdxsToBeDeleted());
        Assert.equals(externalId, params.getExternalId());
        Assert.equals(uid, params.getUid());
    }

    @Test
    public void deleteFlightReminders() {
        AirportManager airportManager = Mockito.mock(AirportManager.class, (Answer) invocation -> Option.none());

        DateTime now = DateTime.now(MoscowTime.TZ);

        MailFlightData.Builder builder = MailFlightData.builder()
                .setFlightNumber("SU2030").setDepartureCity("Москва").setArrivalCity("Хошимин")
                .setDepartureTz(MoscowTime.TZ)
                .setPlannedArrivalDateTime(now);

        ListF<MailFlightData> flightDatas = Cf.list(
                builder.setDepartureDateTime(now.minusMonths(2).toLocalDateTime()).build(),
                builder.setDepartureDateTime(now.plusHours(1).toLocalDateTime()).build(),
                builder.setDepartureDateTime(now.plusMonths(2).toLocalDateTime()).build());

        ListF<FlightEventMeta> flights = flightDatas.map(
                flight -> FlightDataConverter.toFlightEventMeta("mid", airportManager, flight));

        String externalId = "ticket-777";
        PassportUid uid = PassportUid.cons(44902531L);
        MailReminderData remindersData = new MailReminderData(
                Option.none(), Option.some(Duration.standardMinutes(-5)),
                Option.none(), Option.some(new Email("a@b")), Option.some(Duration.standardMinutes(-5)));

        ActionInfo actionInfo = new ActionInfo(now.toInstant(), "", Option.none());
        Function0<ListF<Event>> loadEventsF = () -> eventMdao.findEvents(
                EventsFilter.byExternalId(externalId).toMongoQuery(uid, SpecialClientIds.FLIGHT));

        flightReminderManager.createOrUpdateFlightReminders(uid, externalId, remindersData, flights, actionInfo);

        ListF<Event> events = loadEventsF.apply();
        Assert.hasSize(3, events);
        Assert.notEmpty(events.flatMap(Event::getReminders));

        flightReminderManager.deleteFlightReminders(uid, externalId, actionInfo);

        events = loadEventsF.apply();
        Assert.hasSize(3, events);
        Assert.isEmpty(events.flatMap(Event::getReminders));

        Assert.hasSize(1, bazingaOnetimeJobMdao.findByTaskId(DeleteCloudApiFlightsTask.TASK_ID.getId()));

        FlightRemindersInfo info = flightReminderManager.getFlightRemindersInfo(uid, externalId, flights, actionInfo);
        Assert.none(info.getAutoEmailOffset());
        Assert.none(info.getAutoSmsOffset());
        Assert.none(info.getPhone());
        Assert.none(info.getUserSmsOffset());
    }
}
