package ru.yandex.reminders.logic.flight.rasp;

import lombok.val;
import org.joda.time.DateTimeZone;
import org.joda.time.LocalDateTime;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.test.context.support.AnnotationConfigContextLoader;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.test.Assert;
import ru.yandex.reminders.boot.InitContextConfiguration;
import ru.yandex.reminders.logic.flight.airport.Airport;
import ru.yandex.reminders.logic.flight.airport.AirportManager;
import ru.yandex.reminders.util.TestUtils;

@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(
        loader = AnnotationConfigContextLoader.class,
        classes = {InitContextConfiguration.class, RaspClientContextConfiguration.class}
)
@Ignore("SUBBOTNIK-1476")
public class RaspClientTest extends TestUtils {
    @Autowired
    private RaspClient raspClient;

    @Test
    @Ignore
    public void getAirports() {
        val airports = raspClient.getAirports().getAirports();

        // 1). checks amount
        Assert.ge(airports.size(), 3000);

        val svo = airports.find(Airport.isIataF("SVO"));
        Assert.some(svo);

        // 2). checks Шереметьево
        Assert.some(DateTimeZone.forID("Europe/Moscow"), svo.get().getTz());
        Assert.some("Шереметьево", svo.get().getName());
        Assert.some("Moscow Sheremetyevo International Apt", svo.get().getNameEn());

        // 3). checks presence of at least 1 unique field
        ListF<Airport> airportsWithoutIataAndName = Cf.arrayList();
        for (Airport airport : airports) {
            if (airport.getIata().isEmpty() && airport.getName().isEmpty()) {
                airportsWithoutIataAndName.add(airport);
            }
        }
        if (airportsWithoutIataAndName.isNotEmpty()) {
            Assert.fail(airportsWithoutIataAndName.size()
                    + " airports without iata and name found:" + airportsWithoutIataAndName);
        }

        // 4). checks uniqueness of unique fields
        val notUniqueAirportsByIataAndName =
                airports.groupBy(Airport.getUniqueFieldsF()).filterValues(Cf.List.sizeF().andThen(Cf.Integer.gtF(1)));
        if (notUniqueAirportsByIataAndName.isNotEmpty()) {
            Assert.fail(notUniqueAirportsByIataAndName.size()
                    + " groups of airports with not unique iata+name:" + notUniqueAirportsByIataAndName);
        }

        // 5). checks that AirportManager (and cache) can be built
        val airportManager = new AirportManager(airports);

        // 6). check airportManager
        val now = LocalDateTime.now();
        // for RU
        Assert.equals(DateTimeZone.forID("Europe/Moscow"),
                airportManager.chooseTimezoneForCityAndAirportName("Москва", Option.of("Шереметьево"), now));
        Assert.equals(DateTimeZone.forID("Europe/Moscow"),
                airportManager.chooseTimezoneForCityAndAirportName("Москва", Option.empty(), now));
        Assert.equals(DateTimeZone.forID("Europe/Moscow"),
                airportManager.chooseTimezoneForCityAndAirportName("Москва", Option.of("non-existent"), now));
        // for EN
        Assert.equals(DateTimeZone.forID("Europe/Moscow"),
                airportManager.chooseTimezoneForCityAndAirportName(
                        "Moscow", Option.of("Moscow Sheremetyevo International Apt"), now));
        Assert.equals(DateTimeZone.forID("Europe/Moscow"),
                airportManager.chooseTimezoneForCityAndAirportName("Moscow", Option.empty(), now));
        Assert.equals(DateTimeZone.forID("Europe/Moscow"),
                airportManager.chooseTimezoneForCityAndAirportName("Moscow", Option.of("non-existent"), now));

    }
}
