package ru.yandex.reminders.logic.flight.shift;

import org.bson.types.ObjectId;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.test.context.support.AnnotationConfigContextLoader;

import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.test.Assert;
import ru.yandex.reminders.mongodb.TestMdaoContextConfiguration;
import ru.yandex.reminders.util.TestUtils;

@ContextConfiguration(classes = TestMdaoContextConfiguration.class, loader = AnnotationConfigContextLoader.class)
@RunWith(SpringJUnit4ClassRunner.class)
@Ignore("SUBBOTNIK-1476")
public class FlightShiftMdaoTest extends TestUtils {

    @Autowired
    private FlightShiftMdao flightShiftMdao;

    @Before
    public void before() {
        flightShiftMdao.dropAndCreateCollection();
    }

    @Test
    public void saveAndFindAndUpdateTheSameShift() {
        final String num = "FL213";
        final int geoId = 324123;
        final DateTimeZone tz1 = DateTimeZone.forID("Europe/Moscow");
        final DateTimeZone tz2 = DateTimeZone.forID("+04:00");
        final Instant plannedTs = Instant.now();
        final Instant actualTs = plannedTs.plus(Duration.standardHours(3));

        Assert.equals(0L, flightShiftMdao.count());

        ObjectId id = flightShiftMdao.insertOrUpdate(new FlightShift(num, geoId, tz1, plannedTs, actualTs)).getId();
        Assert.equals(1L, flightShiftMdao.count());

        FlightShift shift = findAndCheckFields(num, geoId, tz1, plannedTs, actualTs, true, id);
        Assert.none(shift.getSendTs());

        ObjectId id2 = flightShiftMdao.insertOrUpdate(new FlightShift(num, geoId, tz2, plannedTs, actualTs)).getId();
        Assert.equals(id, id2);
        Assert.equals(1L, flightShiftMdao.count());

        shift = findAndCheckFields(num, geoId, tz1, plannedTs, actualTs, true, id);
        Assert.none(shift.getSendTs());

        final Instant sendTs = plannedTs.plus(Duration.standardMinutes(15));
        flightShiftMdao.updateSendTs(id, sendTs);
        Assert.equals(1L, flightShiftMdao.count());

        FlightShift updatedShift = findAndCheckFields(num, geoId, tz1, plannedTs, actualTs, true, id);
        Assert.some(sendTs, updatedShift.getSendTs());
    }

    @Test
    public void saveAndFindAndUpdateDifferentShiftsForSameFlight() {
        final String num = "FL213";
        final int geoId = 324123;
        final DateTimeZone tz1 = DateTimeZone.forID("Europe/Moscow");
        final DateTimeZone tz2 = DateTimeZone.forID("+04:00");
        final Instant plannedTs = Instant.now();
        final Instant actualTs = plannedTs.plus(Duration.standardHours(3));

        Assert.equals(0L, flightShiftMdao.count());

        ObjectId id1 = flightShiftMdao.insertOrUpdate(new FlightShift(num, geoId, tz1, plannedTs, actualTs)).getId();
        Assert.equals(1L, flightShiftMdao.count());

        FlightShift shift1 = findAndCheckFields(num, geoId, tz1, plannedTs, actualTs, true, id1);
        Assert.none(shift1.getSendTs());

        ObjectId id2 = flightShiftMdao.insertOrUpdate(
                new FlightShift(num, geoId, tz2, plannedTs, actualTs.plus(Duration.standardMinutes(5)))).getId();
        Assert.notEquals(id1, id2);
        Assert.equals(2L, flightShiftMdao.count());

        shift1 = findAndCheckFields(num, geoId, tz1, plannedTs, actualTs, false, id1);
        Assert.none(shift1.getSendTs());

        ObjectId id3 = flightShiftMdao.insertOrUpdate(
                new FlightShift(num, geoId, tz1, plannedTs, actualTs.plus(Duration.standardMinutes(10)))).getId();
        Assert.notEquals(id2, id3);
        Assert.notEquals(id1, id3);
        Assert.equals(3L, flightShiftMdao.count());

        final Instant sendTs = plannedTs.plus(Duration.standardMinutes(15));
        flightShiftMdao.updateSendTs(id1, sendTs);
        Assert.equals(3L, flightShiftMdao.count());

        FlightShift updatedShift1 = findAndCheckFields(num, geoId, tz1, plannedTs, actualTs, false, id1);
        Assert.some(sendTs, updatedShift1.getSendTs());

        FlightShift shift2 = findAndCheckFields(num, geoId, tz2, plannedTs,
                actualTs.plus(Duration.standardMinutes(5)), false, id2);
        Assert.none(shift2.getSendTs());

        FlightShift shift3 = findAndCheckFields(num, geoId, tz1, plannedTs,
                actualTs.plus(Duration.standardMinutes(10)), true, id3);
        Assert.none(shift3.getSendTs());
    }

    private FlightShift findAndCheckFields(
            String expectedNum, int expectedGeoId, DateTimeZone expectedTz,
            Instant expectedPlannedTs, Instant expectedActualTs, boolean expectedLatest, ObjectId idToFind)
    {
        Option<FlightShift> shiftO = flightShiftMdao.find(idToFind);
        Assert.some(shiftO);

        FlightShift shift = shiftO.get();
        checkFields(expectedNum, expectedGeoId, expectedTz, expectedPlannedTs, expectedActualTs, expectedLatest, shift);
        return shift;
    }

    private void checkFields(
            String expectedNum, int expectedGeoId, DateTimeZone expectedTz,
            Instant expectedPlannedTs, Instant expectedActualTs, boolean expectedLatest, FlightShift actualFlightShift)
    {
        Assert.equals(expectedNum, actualFlightShift.getFlightNum());
        Assert.equals(expectedGeoId, actualFlightShift.getGeoId());
        Assert.equals(expectedTz, actualFlightShift.getTz());
        Assert.equals(expectedPlannedTs, actualFlightShift.getPlannedTs());
        Assert.equals(expectedPlannedTs.toDateTime(expectedTz), actualFlightShift.getPlannedDateTime());
        Assert.equals(expectedActualTs, actualFlightShift.getActualTs());
        Assert.equals(expectedActualTs.toDateTime(expectedTz), actualFlightShift.getActualDateTime());
        Assert.equals(expectedLatest, actualFlightShift.isLatest());
    }

}
