package ru.yandex.reminders.logic.reminder;

import com.mongodb.client.model.Filters;
import org.bson.types.ObjectId;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.test.context.support.AnnotationConfigContextLoader;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.test.Assert;
import ru.yandex.reminders.logic.event.EventId;
import ru.yandex.reminders.mongodb.TestMdaoContextConfiguration;
import ru.yandex.reminders.util.TestUtils;

@ContextConfiguration(classes = TestMdaoContextConfiguration.class, loader = AnnotationConfigContextLoader.class)
@RunWith(SpringJUnit4ClassRunner.class)
@Ignore("SUBBOTNIK-1476")
public class SendResultMdaoTest extends TestUtils {

    @Autowired
    private SendResultMdao sendResultMdao;

    @Before
    public void before() {
        sendResultMdao.dropAndCreateCollection();
    }

    @Test
    public void saveAndFind() {
        EventId eventId = new EventId(PassportUid.cons(12700), "sendResultMdaoTest");
        ObjectId sendReminderId = ObjectId.get();
        ObjectId failReminderId = ObjectId.get();

        sendResultMdao.saveSendResult(SendResult.sent(
                eventId, sendReminderId, Instant.now(), Channel.SMS, Option.some("sentMessageId"), Instant.now()));
        sendResultMdao.saveSendResult(SendResult.failed(
                eventId, failReminderId, Instant.now(), Channel.SMS, "failureReason", Instant.now()));

        ListF<SendResult> sent = sendResultMdao.find(Filters.eq("status", SendResultStatus.SENT.value()));
        ListF<SendResult> failed = sendResultMdao.find(Filters.eq("status", SendResultStatus.FAILED.value()));

        Assert.equals(sendReminderId, sent.single().getReminderId());
        Assert.equals(failReminderId, failed.single().getReminderId());
    }

    @Test
    public void countSentAndFailed() {
        EventId eventId = new EventId(PassportUid.cons(12700), "clientId");

        Instant today = Instant.now();
        Instant yesterday = today.minus(Duration.standardDays(1));

        sendResultMdao.saveSendResult(SendResult.sent(
                eventId, ObjectId.get(), yesterday, Channel.SMS, Option.some("sentId"), yesterday));
        sendResultMdao.saveSendResult(SendResult.failed(
                eventId, ObjectId.get(), yesterday, Channel.SMS, "failReason", yesterday));

        sendResultMdao.saveSendResult(SendResult.sent(
                eventId, ObjectId.get(), today, Channel.SMS, Option.some("sentId"), today));
        sendResultMdao.saveSendResult(SendResult.failed(
                eventId, ObjectId.get(), today, Channel.SMS, "skippedFailReason", today));

        ListF<SendDailyStat> stats = sendResultMdao.countSentAndFailedRemindersGroupedByProcessDate(
                Option.<LocalDate>none(), Option.<LocalDate>none(),
                Option.<String>none(), Option.<Channel>none(), Cf.<String>list());

        Assert.hasSize(2, stats);
        Assert.forAll(stats, SendDailyStat.getSentF().andThenEquals(1));
        Assert.forAll(stats, SendDailyStat.getFailedF().andThenEquals(1));

        LocalDate todayDate = new LocalDate(today, DateTimeZone.UTC);
        stats = sendResultMdao.countSentAndFailedRemindersGroupedByProcessDate(
                Option.some(todayDate), Option.some(todayDate),
                Option.some("clientId"), Option.some(Channel.SMS), Cf.list("skippedFailReason"));

        Assert.hasSize(1, stats);
        Assert.equals(todayDate, stats.single().getDate());
        Assert.equals(1, stats.single().getSent());
        Assert.equals(0, stats.single().getFailed());
    }

}
