package ru.yandex.reminders.logic.sup;

import lombok.val;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.json.JSONObject;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.test.context.ContextConfiguration;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.test.Assert;
import ru.yandex.oauth.OAuth;
import ru.yandex.reminders.api.reminder.ChannelsData;
import ru.yandex.reminders.boot.InitContextConfiguration;
import ru.yandex.reminders.logic.callmeback.in.CallbackRequest;
import ru.yandex.reminders.logic.event.Event;
import ru.yandex.reminders.logic.event.EventData;
import ru.yandex.reminders.logic.event.EventId;
import ru.yandex.reminders.logic.reminder.Origin;
import ru.yandex.reminders.logic.reminder.Reminder;
import ru.yandex.reminders.util.TestUtils;

import java.io.File;
import java.io.IOException;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import static org.apache.commons.io.FileUtils.readFileToString;

@ContextConfiguration(
        classes = InitContextConfiguration.class
)
public class SupPushManagerTest extends TestUtils {
    private final SupPushClient supClient = new SupPushClient(
            "https://push-beta.n.yandex-team.ru/pushes?dry_run=1", 1, Timeout.timeout(5, TimeUnit.SECONDS));
    private final XivaPushClient xivaClient = new XivaPushClient(
            "https://push-sandbox.yandex.ru/v2/send", 1, Timeout.timeout(5, TimeUnit.SECONDS));
    private final String clientId = "yandex-test";
    @Autowired
    OAuth oauth;
    String oauthClientId = "f63c3a60967a47a6917b4e2f97912ea9";
    String oauthClientPassword = "730869deb9ba44abb58a51c7768db2c1";
    private SupClientSettingsRegistry registry;
    private SupPushManager manager;
    @Value("${sup.push.token}")
    private String supTokenPath;

    @Value("${xiva.push.token}")
    private String xivaTokenPath;

    @Before
    public void setup() throws IOException {
        registry = Mockito.mock(SupClientSettingsRegistry.class);
        manager = new SupPushManager(supClient, xivaClient, registry);

        String xivaToken = "";
        String supToken = "";

        if (xivaTokenPath.isEmpty()) {
            //We use only first oauth token
            val oauthToken = oauth.getOauthToken(oauthClientId, oauthClientPassword).get(0);
            val httpClient = HttpClients.createDefault();

            val get = new HttpGet("https://vault-api.passport.yandex.net/1/versions/ver-01csyz2dsg6ays76s5cmqzpspq/");
            get.addHeader("Authorization", "OAuth " + oauthToken);
            val res = httpClient.execute(get);
            val obj = new JSONObject(EntityUtils.toString(res.getEntity()));
            xivaToken = obj.getJSONObject("version").getJSONArray("value").getJSONObject(0).getString("value");
        } else {
            xivaToken = readFileToString(new File(xivaTokenPath)).trim();
        }

        if (supTokenPath.isEmpty()) {
            //We use only first oauth token
            supToken = oauth.getOauthToken(oauthClientId, oauthClientPassword).get(0);
        } else {
            supToken = readFileToString(new File(supTokenPath)).trim();
        }

        val overrides = new SupPushRequest();
        overrides.setTtl(Option.of(3600));
        overrides.setProject(Option.of("sup"));

        val recipientExpression = "tag:uid==<uid> AND app_id IN ('ru.yandex.mobile', 'ru.yandex.mobile.inhouse')";

        Mockito.when(registry.getO(Mockito.eq(clientId))).thenReturn(Option.of(new SupClientSettings(
                clientId, supToken, recipientExpression, "PERSONAL", overrides,
                Option.of(xivaToken))));
    }

    @Test
    public void sentSup() {
        val reminder = Reminder.sup(
                DateTime.now(DateTimeZone.UTC), Option.of("Яндекс"),
                Option.of("Узнайте в Яндексе"), Option.of("dialog://?directives=JSON"),
                new ChannelsData.Sup.SupExtraFields());


        val status = manager.pushSup(CallbackRequest.create(consEvent(reminder), reminder));

        if (status.isTryAgain()) {
            Assert.fail(status.asTryAgain().getMessage());
        }
        if (status.isFailed()) {
            Assert.fail(status.asFailed().getFailureReason());
        }
        Assert.some(status.asSent().getMessageId());
    }

    @Test
    public void sentSupWithDeviceId() {
        Reminder reminder = Reminder.sup(
                DateTime.now(DateTimeZone.UTC), Option.of("Яндекс"),
                Option.of("Узнайте в Яндексе"), Option.of("dialog://?directives=JSON"),
                new ChannelsData.Sup.SupExtraFields(Optional.of("testdid"), Optional.empty(), Optional.empty()));

        val status = manager.pushSup(CallbackRequest.create(consEvent(reminder), reminder));

        if (status.isTryAgain()) {
            Assert.fail(status.asTryAgain().getMessage());
        }
        if (status.isFailed()) {
            Assert.fail(status.asFailed().getFailureReason());
        }
        Assert.some(status.asSent().getMessageId());
    }

    @Test
    public void sentXiva() {
        val reminder = Reminder.xiva(DateTime.now(DateTimeZone.UTC), Origin.AUTO, "dialog://?directives=JSON");

        Assert.isTrue(manager.pushXiva(consEvent(reminder).getId(), reminder).isSent());
    }

    private Event consEvent(Reminder reminder) {
        return new Event(
                new EventId(PassportUid.cons(44902531), clientId),
                EventData.empty().withReminders(Cf.list(reminder)), Option.empty(), Instant.now(), "req-id");
    }
}
