package ru.yandex.reminders.util.task;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDateTime;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.test.context.support.AnnotationConfigContextLoader;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.test.Assert;
import ru.yandex.reminders.api.flight.FlightDataConverter;
import ru.yandex.reminders.api.flight.MailReminderData;
import ru.yandex.reminders.api.unistat.UnistatConfiguration;
import ru.yandex.reminders.logic.event.*;
import ru.yandex.reminders.logic.flight.FlightCity;
import ru.yandex.reminders.logic.flight.FlightEventMeta;
import ru.yandex.reminders.logic.flight.FlightReminderManager;
import ru.yandex.reminders.logic.flight.FlightReminderManagerContextConfiguration;
import ru.yandex.reminders.logic.flight.shift.FlightShift;
import ru.yandex.reminders.logic.flight.shift.FlightShiftMdao;
import ru.yandex.reminders.logic.reminder.Channel;
import ru.yandex.reminders.logic.update.ActionInfo;
import ru.yandex.reminders.logic.user.UserContextConfiguration;
import ru.yandex.reminders.mongodb.BazingaMongoClientContextConfiguration;
import ru.yandex.reminders.mongodb.BazingaOnetimeJobMdao;
import ru.yandex.reminders.mongodb.TestMdaoContextConfiguration;
import ru.yandex.reminders.tvm.TvmClientTestConfiguration;
import ru.yandex.reminders.util.TestUtils;

import static ru.yandex.reminders.util.DateTimeUtilsTestCommon.transitedDate;
import static ru.yandex.reminders.util.DateTimeUtilsTestCommon.tzFrom;
import static ru.yandex.reminders.util.DateTimeUtilsTestCommon.tzTo;

@ContextConfiguration(loader = AnnotationConfigContextLoader.class, classes = {
        TvmClientTestConfiguration.class,
        BazingaMongoClientContextConfiguration.class,
        TestMdaoContextConfiguration.class,
        EventManagerTestContextConfiguration.class,
        FlightReminderManagerContextConfiguration.class,
        UserContextConfiguration.class,
        ManualTaskContextConfiguration.class,
        UnistatConfiguration.class,
})
@RunWith(SpringJUnit4ClassRunner.class)
@Ignore("SUBBOTNIK-1476")
public class TimeZoneMigrationTaskTest extends TestUtils {
    @Autowired
    private EventManager eventManager;
    @Autowired
    private EventMdao eventMdao;
    @Autowired
    private FlightReminderManager flightReminderManager;
    @Autowired
    private FlightShiftMdao flightShiftMdao;
    @Autowired
    private TimeZoneMigrationTask timeZoneMigrationTask;
    @Autowired
    private BazingaOnetimeJobMdao bazingaOnetimeJobMdao;

    @Before
    public void before() {
        eventMdao.dropAndCreateCollection();
        flightShiftMdao.dropAndCreateCollection();
        bazingaOnetimeJobMdao.removeAll();
    }

    @Test
    public void test() {
        PassportUid uid = new PassportUid(132225125);
        Email email = new Email("calendartestuser@yandex.ru");

        LocalDateTime planedDepartureDateTime = transitedDate.toDateTime(tzFrom).plusDays(2).toLocalDateTime();
        LocalDateTime departureDateTime = planedDepartureDateTime.plusHours(1);
        LocalDateTime shiftedDepartureDateTime = planedDepartureDateTime.plusHours(2);

        Duration reminderOffset = Duration.standardDays(-1);

        Function<LocalDateTime, LocalDateTime> translateToOldMoscowTimeF =
                t -> new LocalDateTime(t.toDateTime(tzFrom), tzTo);

        MailReminderData reminderData = new MailReminderData(
                Option.none(), Option.some(reminderOffset),
                Option.none(), Option.some(email), Option.some(reminderOffset));

        FlightEventMeta flightData = new FlightEventMeta(
                "mid31829", "UT265", Option.none(),
                Option.some(planedDepartureDateTime).map(translateToOldMoscowTimeF),
                new FlightCity("Москва"), Option.none(),
                translateToOldMoscowTimeF.apply(departureDateTime), tzTo,
                new FlightCity("Курган"), Option.none(),
                Option.none(), Option.none(),
                Option.none(), Option.none(), Option.none(),
                Option.none(), Option.none(), Option.none(),
                Option.none(), Option.none(), Option.none());

        flightData.setNotMigrated();

        EventData flightEventData = new EventData(
                Option.some(ru.yandex.reminders.api.reminder.Source.INTERNAL),
                Option.none(), Option.none(), Option.none(),
                FlightDataConverter.toReminders(flightData, reminderData, Instant.now()), Option.some(flightData));

        ActionInfo actionInfo = new ActionInfo(
                planedDepartureDateTime.toDateTime(tzFrom).plus(reminderOffset).toInstant(), "req-id", Option.none());

        Event event = eventManager.createOrUpdateEvent(
                new EventId(uid, SpecialClientIds.FLIGHT, "external-id"), flightEventData, actionInfo);

        FlightShift shift = new FlightShift("UT265", 15, tzTo,
                translateToOldMoscowTimeF.apply(planedDepartureDateTime),
                translateToOldMoscowTimeF.apply(shiftedDepartureDateTime));

        shift.setNotMigrated();

        shift = flightShiftMdao.insertOrUpdate(shift);
        Instant shiftSendTs = flightReminderManager.scheduleFlightShiftSmsSending(shift, actionInfo);

        flightShiftMdao.updateSendTs(shift.getId(), shiftSendTs);

        timeZoneMigrationTask.doExecute();

        event = eventManager.findEvent(event.getId()).get();
        shift = flightShiftMdao.find(shift.getId()).get();

        OnetimeJob reminderJob = bazingaOnetimeJobMdao.findByActiveUniqueId(
                BazingaOnetimeJobMdao.reminderIdToActiveUniqueId(event.getReminders().first().getId())).single();
        OnetimeJob shiftJob = bazingaOnetimeJobMdao.findByActiveUniqueId(
                BazingaOnetimeJobMdao.flightShiftIdToActiveUniqueId(shift.getId())).single();

        Instant planedDepartureTs = planedDepartureDateTime.toDateTime(tzTo).toInstant();
        Instant departureTs = departureDateTime.toDateTime(tzTo).toInstant();

        Instant shiftedDepartureTs = shiftedDepartureDateTime.toDateTime(tzTo).toInstant();
        Instant shiftedSendTs = new LocalDateTime(shiftSendTs, tzFrom).toDateTime(tzTo).toInstant();

        Assert.isTrue(event.getFlightMeta().get().isMigrated());
        Assert.isTrue(shift.isMigrated());

        Assert.some(event.getFlightMeta());
        Assert.equals(departureDateTime, event.getFlightMeta().get().getDepartureLocalDateTime());

        Assert.some(planedDepartureTs, event.getFlightMeta().get().getPlannedDepartureTs());
        Assert.forAll(event.getReminders().filterNot(r -> r.getChannel() == Channel.CLOUD_API),
                r -> r.getSendTs().isEqual(departureTs.plus(reminderOffset)));

        Assert.equals(planedDepartureTs, shift.getPlannedTs());
        Assert.equals(shiftedDepartureTs, shift.getActualTs());
        Assert.some(shiftedSendTs, shift.getSendTs());

        Assert.equals(departureTs.plus(reminderOffset), reminderJob.getScheduleTime());
        Assert.equals(shift.getSendTs().get(), shiftJob.getScheduleTime());
    }
}
