package ru.yandex.reminders.api;

import lombok.val;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import ru.yandex.commune.a3.action.invoke.ActionInvocationContext;
import ru.yandex.commune.a3.action.parameter.convert.ConverterToType;
import ru.yandex.commune.a3.action.result.ApplicationInfo;
import ru.yandex.commune.a3.action.result.ApplicationResultType;
import ru.yandex.commune.a3.action.result.type.MediaType;
import ru.yandex.misc.annotation.Nullable;
import ru.yandex.misc.version.AppName;
import ru.yandex.misc.version.Version;
import ru.yandex.misc.web.servlet.PingConfigurableServlet;
import ru.yandex.misc.web.servletContainer.SingleWarJetty;
import ru.yandex.reminders.api.a3.bind.JsonActionResultSerializer;
import ru.yandex.reminders.api.a3.bind.JsonBinder;
import ru.yandex.reminders.api.a3.interceptors.LogActionParametersInterceptor;
import ru.yandex.reminders.api.a3.interceptors.MongoDbAvailabilityInterceptor;
import ru.yandex.reminders.api.a3.interceptors.RequestHandlesInterceptor;
import ru.yandex.reminders.api.calendar.ApiCalendarContextConfiguration;
import ru.yandex.reminders.api.callmeback.ApiCallmebackContextConfiguration;
import ru.yandex.reminders.api.flight.ApiFlightContextConfiguration;
import ru.yandex.reminders.api.reminder.ApiReminderContextConfiguration;
import ru.yandex.reminders.api.tv.ApiTvContextConfiguration;
import ru.yandex.reminders.api.unistat.ApiUnistatContextConfiguration;
import ru.yandex.reminders.boot.RemindersAppContextConfiguration;
import ru.yandex.reminders.log.TskvLogActionInvocationInterceptor;
import ru.yandex.reminders.logic.callmeback.CallmebackManagerContextConfiguration;
import ru.yandex.reminders.logic.flight.FlightReminderManagerContextConfiguration;
import ru.yandex.reminders.logic.sup.SupClientContextConfiguration;
import ru.yandex.reminders.logic.tv.TvReminderManagerContextConfiguration;
import ru.yandex.reminders.tvm.TvmClientConfiguration;
import ru.yandex.reminders.util.DateTimeZoneConverter;

@Configuration
@Import({TvmClientConfiguration.class,
    RemindersAppContextConfiguration.class,
    FlightReminderManagerContextConfiguration.class,
    TvReminderManagerContextConfiguration.class,
    ApiFlightContextConfiguration.class,
    ApiTvContextConfiguration.class,
    ApiCalendarContextConfiguration.class,
    ApiCallmebackContextConfiguration.class,
    CallmebackManagerContextConfiguration.class,
    ApiReminderContextConfiguration.class,
    SupClientContextConfiguration.class,
    ApiUnistatContextConfiguration.class,
})
public class ApiAppContextConfiguration {
    @Bean
    public SingleWarJetty apiJetty(
            @Value("${api.jetty.port}") int port,
            @Value("${api.jetty.maxThreads}") int maxThreads)
    {
        val jetty = new SingleWarJetty();
        jetty.setHttpPort(port);
        jetty.setMaxThreads(maxThreads);
        jetty.setLookupServletsInContext(false);

        jetty.addServletMapping("/ping/*", apiPingConfigurableServlet());

        return jetty;
    }

    @Bean
    public PingConfigurableServlet apiPingConfigurableServlet() {
        return new PingConfigurableServlet();
    }

    @Bean
    public ApplicationInfo applicationInfo(AppName appName, Version version) {
        return new ApplicationInfo(appName.appName(), version.getProjectVersion());
    }

    @Bean
    public ApplicationResultType applicationResultType() {
        return new ApplicationResultType(MediaType.APPLICATION_JSON);
    }

    @Bean
    public JsonBinder jsonBinder() {
        return new JsonBinder(ApiBender.mapper);
    }

    @Bean
    public JsonActionResultSerializer jsonActionResultSerializer() {
        return new JsonActionResultSerializer(ApiBender.mapper);
    }

    @Bean
    public RequestHandlesInterceptor requestHandlesInterceptor() {
        return new RequestHandlesInterceptor();
    }

    @Bean
    public LogActionParametersInterceptor logActionParametersInterceptor() {
        return new LogActionParametersInterceptor();
    }

    @Bean
    public TskvLogActionInvocationInterceptor tskvLogActionInvocationInterceptor() {
        return new TskvLogActionInvocationInterceptor();
    }

    @Bean
    public MongoDbAvailabilityInterceptor mongoDbAvailabilityInterceptor() {
        return new MongoDbAvailabilityInterceptor();
    }

    @Bean
    public ConverterToType converterToDateTime() {
        // TODO: remove this ugly ConverterToType wrapper around dateTimeWithTzConverter when iceberg will be fixed
        // (see: iceberg rev: 1264b95c5f92)
        return new ConverterToType<DateTime>() {
            public DateTime convert(@Nullable String s, ActionInvocationContext invocationContext) {
                return DateTimeConverters.dateTimeWithTzConverter.convert(s, invocationContext);
            }
        };
    }

    @Bean
    public ConverterToType converterToDateTimeZone() {
        return new DateTimeZoneConverter();
    }
}
