package ru.yandex.reminders.api;

import lombok.val;
import org.bson.types.ObjectId;
import org.joda.time.*;

import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonValue;
import ru.yandex.commune.json.bender.JsonValueMarshaller;
import ru.yandex.commune.json.bender.JsonValueUnmarshaller;
import ru.yandex.commune.mongo3.bender.MongoObjectIdMarshaller;
import ru.yandex.commune.mongo3.bender.MongoObjectIdUnmarshaller;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.config.*;
import ru.yandex.misc.bender.custom.InstantAsMillisMarshaller;
import ru.yandex.misc.bender.custom.InstantAsMillisUnmarshaller;
import ru.yandex.misc.bender.parse.simpleType.SimpleTypeUnmarshallerSupport;
import ru.yandex.misc.bender.serialize.BenderJsonWriter;
import ru.yandex.misc.bender.serialize.simpleType.SimpleTypeMarshallerSupport;
import ru.yandex.misc.bender.serialize.simpleType.StringValueMarshaller;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.email.bender.EmailMarshaller;
import ru.yandex.misc.email.bender.EmailUnmarshaller;
import ru.yandex.reminders.mongodb.DateTimeZoneMarshaller;
import ru.yandex.reminders.mongodb.DateTimeZoneUnmarshaller;
import ru.yandex.reminders.util.ValueOrObjectMarshallerUnmarshallerFactory;

public class ApiBender {
    public static final BenderMapper mapper = new BenderMapper(getConfiguration());

    public static BenderConfiguration getConfiguration() {
        val muf = new ValueOrObjectMarshallerUnmarshallerFactory();

        val settings = new BenderSettings(
                MembersToBind.WITH_ANNOTATIONS, new JsonSettings(false, false, false, JsonLongFormat.NUMBER));

        return new BenderConfiguration(settings,
                CustomMarshallerUnmarshallerFactoryUtils.combine(muf, CustomMarshallerUnmarshallerFactoryBuilder.cons()
                        .add(Duration.class, new DurationAsMinutesMarshaller(), new DurationAsMinutesUnmarshaller())
                        .add(Instant.class, new InstantAsMillisMarshaller(), new InstantAsMillisUnmarshaller())
                        .add(LocalDateTime.class, new LocalDateTimeMarshaller(), new LocalDateTimeUnmarshaller())
                        .add(DateTime.class, new DateTimeMarshaller(), new DateTimeUnmarshaller())
                        .add(Email.class, new EmailMarshaller(), new EmailUnmarshaller())
                        .add(DateTimeZone.class, new DateTimeZoneMarshaller(), new DateTimeZoneUnmarshaller())
                        .add(JsonValue.class, new JsonValueMarshaller(), new JsonValueUnmarshaller())
                        .add(JsonObject.class, new JsonValueMarshaller(), new JsonValueUnmarshaller())
                        .add(ObjectId.class, new MongoObjectIdMarshaller(), new MongoObjectIdUnmarshaller())
                        .build()));
    }

    private static class DurationAsMinutesMarshaller extends SimpleTypeMarshallerSupport {
        @Override
        protected String toStringValueForXml(Object o) {
            return Long.toString(((Duration) o).getStandardMinutes());
        }

        @Override
        protected void writeJson(BenderJsonWriter json, Object o) {
            json.writeNumber(((Duration) o).getStandardMinutes());
        }
    }

    private static class DurationAsMinutesUnmarshaller extends SimpleTypeUnmarshallerSupport {
        protected Duration convert(String o) {
            return Duration.standardMinutes(Long.parseLong(o));
        }
    }

    private static class LocalDateTimeMarshaller extends StringValueMarshaller {
        protected String toStringValue(Object o) {
            return DateTimeConverters.localDateTimeConverter.toString((LocalDateTime) o);
        }
    }

    private static class LocalDateTimeUnmarshaller extends SimpleTypeUnmarshallerSupport {
        protected Object convert(String o) {
            return DateTimeConverters.localDateTimeConverter.parse(o);
        }
    }

    private static class DateTimeMarshaller extends StringValueMarshaller {
        protected String toStringValue(Object o) {
            return DateTimeConverters.dateTimeWithTzConverter.toString((DateTime) o);
        }
    }

    private static class DateTimeUnmarshaller extends SimpleTypeUnmarshallerSupport {
        protected Object convert(String o) {
            return DateTimeConverters.raspOrIsoDateTimeWithTzConverter.apply(o);
        }
    }
}
