package ru.yandex.reminders.api;

import org.joda.time.DateTime;
import org.joda.time.LocalDateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.misc.time.TimeFormats;
import ru.yandex.reminders.util.DateTimeConverter;

public class DateTimeConverters {

    public static final DateTimeConverter<LocalDateTime> localDateTimeConverter =
            DateTimeConverter.localDateTimeConverter(TimeFormats.ISO_DATE_TIME_FORMATTER);

    public static final DateTimeConverter<DateTime> dateTimeWithTzConverter =
            DateTimeConverter.dateTimeConverter(ISODateTimeFormat.dateTimeNoMillis().withOffsetParsed());

    public static final DateTimeConverter<DateTime> raspDateTimeWithTzConverter =
            DateTimeConverter.dateTimeConverter(DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss ZZ").withOffsetParsed());

    public static final Function<String, DateTime> raspOrIsoDateTimeWithTzConverter = (s) -> {
        try {
            return raspDateTimeWithTzConverter.parse(s);
        } catch (IllegalArgumentException e) {
            return dateTimeWithTzConverter.parse(s);
        }
    };

    private static final ListF<DateTimeFormatter> hotelDateFormatters = Cf.list(
            DateTimeFormat.forPattern("yyyy-M-d H:m:s"),
            DateTimeFormat.forPattern("d.M.yyyy H:m:s"));

    public static final Function<String, LocalDateTime> hotelDateParser = (s) -> {
        ListF<LocalDateTime> result = hotelDateFormatters.filterMap(formatter -> {
            try {
                return Option.of(formatter.parseLocalDateTime(s));
            } catch (IllegalArgumentException e) {
                return Option.empty();
            }
        });
        return result.firstO().getOrThrow("Cannot parse date time: ", s);
    };
}