package ru.yandex.reminders.api.a3.bind;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParseException;
import lombok.val;
import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.a3.action.invoke.ActionInvocationContext;
import ru.yandex.commune.a3.action.parameter.Parameter;
import ru.yandex.commune.a3.action.parameter.ParameterDescriptor;
import ru.yandex.commune.a3.action.parameter.WebRequest;
import ru.yandex.commune.a3.action.parameter.bind.ParameterBinder;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.parse.BenderParserException;
import ru.yandex.misc.bender.serialize.BenderJsonGeneratorWrapper;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.reminders.api.a3.interceptors.LoggingParameterBinder;

import java.io.IOException;
import java.io.StringWriter;
import java.nio.charset.Charset;

public class JsonBinder implements ParameterBinder, LoggingParameterBinder {
    private static final Logger logger = LoggerFactory.getLogger(JsonBinder.class);
    private final BenderMapper mapper;

    public JsonBinder(BenderMapper mapper) {
        this.mapper = mapper;
    }

    @Override
    public Object createAndBind(
            WebRequest webRequest, ActionInvocationContext invocationContext, ParameterDescriptor parameterDescriptor) {
        Class<?> parameterClass = parameterDescriptor.getParameterType().getActualType().asClass().getClazz();

        val request = webRequest.getHttpServletRequest().getInputStreamX();

        try {
            val body = InputStreamSourceUtils.wrap(request).readBytes();
            logger.debug("web request body={}", new String(body, Charset.forName("UTF-8")));
            return mapper.parseJson(parameterClass, body);
        } catch (BenderParserException | IllegalArgumentException e) {
            throw new JsonBinderParseException(e);
        } catch (RuntimeException e) {
            if (e.getCause() instanceof JsonParseException) {
                throw new JsonBinderParseException(e);
            }
            throw e;
        }
    }

    @Override
    public Option<String> getNameForLogging(Parameter parameter) {
        return Option.empty();
    }

    @Override
    public Option<String> getValueForLogging(Parameter parameter) {
        val value = parameter.getValue().get();

        try {
            val wr = new StringWriter();
            val gen = new JsonFactory().createJsonGenerator(wr).useDefaultPrettyPrinter();
            val jw = new BenderJsonGeneratorWrapper(gen);

            mapper.serializeJson(value, jw);

            jw.close();

            return Option.of(wr.toString());
        } catch (IOException e) {
            throw IoUtils.translate(e);
        }
    }
}