package ru.yandex.reminders.api.calendar;

import lombok.val;
import org.joda.time.DateTimeZone;
import org.joda.time.LocalDate;
import org.joda.time.LocalTime;
import org.springframework.beans.factory.annotation.Autowired;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.reminders.logic.event.Event;
import ru.yandex.reminders.logic.event.EventManager;
import ru.yandex.reminders.logic.event.EventsFilter;
import ru.yandex.reminders.logic.event.SpecialClientIds;
import ru.yandex.reminders.logic.flight.FlightReminderManager;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class CalendarEventManager {

    @Autowired
    private EventManager eventManager;
    @Autowired
    private FlightReminderManager flightReminderManager;


    public List<EventInfo> findEvents(PassportUid uid, LocalDate from, LocalDate till, DateTimeZone tz) {
        val searchFrom = from.minusDays(2).toDateTimeAtStartOfDay(tz).toInstant();
        val searchTo = till.plusDays(2).toDateTimeAtStartOfDay(tz).toInstant();

        val events = eventManager.findEventsByReminderSendTs(uid,
                Cf.list(SpecialClientIds.FLIGHT, SpecialClientIds.HOTEL, SpecialClientIds.NEZABUDKA),
                new InstantInterval(searchFrom, searchTo));

        val isFlightF = Event.getClientIdF().andThenEquals(SpecialClientIds.FLIGHT);

        val actualFlights = flightReminderManager.getFlightEventsWithActualMeta(events.filter(isFlightF));
        val otherEvents = events.filter(isFlightF.notF());

        val converted = Stream.concat(actualFlights.stream(), otherEvents.stream())
                .map(EventInfoConverter::convertSafe)
                .flatMap(o -> o.map(Stream::of).orElseGet(Stream::empty));

        val localStartMatchesF = EventInfo.localStartInF(
                from.toLocalDateTime(LocalTime.MIDNIGHT), till.plusDays(1).toLocalDateTime(LocalTime.MIDNIGHT));

        val absoluteStartMatchesF = EventInfo.absoluteStartInF(
                from.toDateTimeAtStartOfDay(tz), till.plusDays(1).toDateTimeAtStartOfDay(tz));

        return converted.filter(absoluteStartMatchesF.orF(localStartMatchesF)).collect(Collectors.toList());
    }

    public void deleteEvent(PassportUid uid, String clientId, String externalId, long idx) {
        eventManager.deleteEvents(uid, clientId, EventsFilter.byExternalId(externalId).andByIdx(Cf.list(idx)));
    }
}
