package ru.yandex.reminders.api.calendar;

import lombok.val;
import org.joda.time.DateTimeZone;
import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.reminders.api.DateTimeConverters;
import ru.yandex.reminders.logic.event.Event;
import ru.yandex.reminders.logic.event.SpecialClientIds;
import ru.yandex.reminders.logic.reminder.Reminder;
import ru.yandex.reminders.util.JsonObjectUtils;

import java.util.Optional;
import java.util.function.Function;

public class EventInfoConverter {

    private static final Logger logger = LoggerFactory.getLogger(EventInfoConverter.class);

    public static Optional<EventInfo> convertSafe(Event event) {
        try {
            return Optional.of(convert(event));
        } catch (Exception e) {
            logger.error("Failed to convert event {}", event.getId(), e);
            return Optional.empty();
        }
    }

    public static EventInfo convert(Event event) {
        if (SpecialClientIds.FLIGHT.equals(event.getClientId())) {
            return convertFlightEvent(event);
        }
        if (SpecialClientIds.HOTEL.equals(event.getClientId())) {
            return convertHotelReservation(event);
        }
        if (SpecialClientIds.NEZABUDKA.equals(event.getClientId())) {
            return convertNezabudkaEvent(event);
        }
        throw new IllegalArgumentException("Unexpected client id " + event.getClientId());
    }

    private static EventInfo convertFlightEvent(Event event) {
        Validate.isTrue(SpecialClientIds.FLIGHT.equals(event.getClientId()));

        val flight = event.getFlightMeta().get();

        val start = flight.getDepartureDateTime();
        val name = "Перелёт " + flight.getDepartureCity().getName() + " – " + flight.getArrivalCity().getName();
        val description = Option.<String>empty();

        return new EventInfo(event.getId(), start, name, description, event.getReminderChannels());
    }

    private static EventInfo convertHotelReservation(Event event) {
        Validate.isTrue(SpecialClientIds.HOTEL.equals(event.getClientId()));

        val data = JsonObjectUtils.getObjectFieldO(event.getData().get(), "data").orElseThrow(() -> new RuntimeException("data"));

        val hotel = JsonObjectUtils.getStringFieldValueO(data, "hotel").orElseThrow(() -> new RuntimeException("hotel"));
        val innDateStr = JsonObjectUtils.getStringFieldValueO(data, "check-inn_date").orElseThrow(() -> new RuntimeException("check-inn_date"));
        val innMs = JsonObjectUtils.getNumberFieldLongValueO(data, "check-inn_timestamp").orElseThrow(() -> new RuntimeException("check-inn_ts"));

        val innDate = DateTimeConverters.hotelDateParser.apply(innDateStr);
        val tz = DateTimeZone.forOffsetMillis((int) (innDate.toDateTime(DateTimeZone.UTC).getMillis() - innMs));

        val start = innDate.toDateTime(tz);
        val name = "Проживание в отеле " + hotel;
        val description = Option.<String>empty();

        return new EventInfo(event.getId(), start, name, description, event.getReminderChannels());
    }

    private static EventInfo convertNezabudkaEvent(Event event) {
        Validate.isTrue(SpecialClientIds.NEZABUDKA.equals(event.getClientId()));

        Validate.hasSize(1, event.getReminders().map(Reminder.getSendDateF()).stableUnique(), "Inconvertible event");
        val start = event.getReminders().first().getSendDate();

        val name = event.getName().getOrElse("Unnamed event");
        val description = event.getDescription();

        return new EventInfo(event.getId(), start, name, description, event.getReminderChannels());
    }
}
