package ru.yandex.reminders.api.flight;

import lombok.val;
import org.joda.time.*;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.regex.Pattern2;
import ru.yandex.reminders.logic.flight.FlightCity;
import ru.yandex.reminders.logic.flight.FlightEventMeta;
import ru.yandex.reminders.logic.flight.FlightItem;
import ru.yandex.reminders.logic.flight.airport.Airport;
import ru.yandex.reminders.logic.flight.airport.AirportManager;
import ru.yandex.reminders.logic.flight.shift.SendingTimeUtils;
import ru.yandex.reminders.logic.reminder.Channel;
import ru.yandex.reminders.logic.reminder.Origin;
import ru.yandex.reminders.logic.reminder.PhoneNumber;
import ru.yandex.reminders.logic.reminder.Reminder;

import java.util.Optional;
import java.util.regex.Pattern;

public class FlightDataConverter {

    private static final Logger logger = LoggerFactory.getLogger(FlightDataConverter.class);

    private static final Duration STANDARD_CLOUD_API_OFFSET = Duration.standardDays(-30);
    private static final Duration STANDARD_PANEL_OFFSET = Duration.standardDays(-1);
    private static final Pattern2 FLIGHT_NUMBER_NORMALIZE_PATTERN = new Pattern2(Pattern.compile("[-\\s]"));

    public static ListF<FlightEventMeta> toFlightEventMetas(
            MailFlightRemindersData mailFlightRemindersData, AirportManager airportManager) {
        return mailFlightRemindersData.getFlights().map(flight -> toFlightEventMeta(
                mailFlightRemindersData.getMid(), airportManager, flight,
                mailFlightRemindersData.getLang(), mailFlightRemindersData.getYaDomain()));
    }

    public static ListF<Reminder> toReminders(FlightEventMeta meta, MailReminderData mailReminderData, Instant now) {
        ListF<Reminder> result = Cf.arrayList();
        if (mailReminderData.getUserSmsOffset().isPresent()) {
            result.add(userSms(meta, mailReminderData.getPhone(), mailReminderData.getUserSmsOffset()));
        } else if (mailReminderData.getAutoSmsOffset().isPresent()) {
            result.add(autoSms(meta, mailReminderData.getPhone(), mailReminderData.getAutoSmsOffset()));
        }
        if (mailReminderData.getAutoEmailOffset().isPresent()) {
            result.add(autoEmail(meta, mailReminderData.getEmail(), mailReminderData.getAutoEmailOffset()));
        }
        result.add(autoPanel(meta));
        calcCloudApiSendDateTime(meta, meta.getDepartureDateTime(), now).map(Reminder::cloudApi).ifPresent(result::add);

        return result;
    }

    public static DateTime calcSendDateTime(DateTime departureDateTime, DateTimeZone departureTz,
                                            Duration offset, Channel channel, Origin origin) {
        LocalDateTime sendTs = departureDateTime.toLocalDateTime().plus(offset);
        if (channel == Channel.SMS && origin == Origin.AUTO) {
            sendTs = SendingTimeUtils.calcFlightReminderAutoSmsSendTs(sendTs);
        }
        return sendTs.toDateTime(departureTz);
    }

    public static Optional<DateTime> calcCloudApiSendDateTime(FlightEventMeta flight, DateTime actualDepartureDateTime, Instant now) {
        val firstSendDateTime = flight.getPlannedDepartureDateTime()
                .getOrElse(flight.getDepartureDateTime()).plus(STANDARD_CLOUD_API_OFFSET);

        if (actualDepartureDateTime.isBefore(now)) {
            return Optional.empty();
        }
        if (firstSendDateTime.isAfter(now)) {
            return Optional.of(firstSendDateTime);
        }
        return Optional.of(now.toDateTime(flight.getDepartureCityTz()));
    }

    public static Reminder userSms(FlightEventMeta meta, Option<PhoneNumber> phone, Option<Duration> offset) {
        return Reminder.sms(
                calcSendDateTime(
                        meta.getDepartureDateTime(), meta.getDepartureCityTz(),
                        offset.get(), Channel.SMS, Origin.USER),
                Option.of(offset.get().toStandardMinutes().getMinutes()),
                Option.of(Origin.USER),
                phone,
                Option.empty()
        );
    }

    public static Reminder autoSms(FlightEventMeta meta, Option<PhoneNumber> phone, Option<Duration> offset) {
        return Reminder.sms(
                calcSendDateTime(
                        meta.getDepartureDateTime(), meta.getDepartureCityTz(),
                        offset.get(), Channel.SMS, Origin.AUTO),
                Option.of(offset.get().toStandardMinutes().getMinutes()),
                Option.of(Origin.AUTO),
                phone,
                Option.empty()
        );
    }

    public static Reminder autoEmail(FlightEventMeta meta, Option<Email> email, Option<Duration> offset) {
        return Reminder.email(
                calcSendDateTime(
                        meta.getDepartureDateTime(), meta.getDepartureCityTz(),
                        offset.get(), Channel.EMAIL, Origin.AUTO),
                Option.of(offset.get().toStandardMinutes().getMinutes()),
                Option.of(Origin.AUTO),
                Option.empty(),
                email,
                Option.empty(),
                Option.empty(),
                Option.empty()
        );
    }

    public static Reminder autoPanel(FlightEventMeta meta) {
        DateTime remindDate = calcSendDateTime(
                meta.getDepartureDateTime(), meta.getDepartureCityTz(),
                STANDARD_PANEL_OFFSET, Channel.PANEL, Origin.AUTO);

        return Reminder.panel(remindDate, Option.none(), Option.none(), Option.none());
    }

    public static FlightEventMeta toFlightEventMeta(String mid, AirportManager airportManager, MailFlightData flight) {
        return toFlightEventMeta(mid, airportManager, flight, Option.none(), Option.none());
    }

    public static FlightEventMeta toFlightEventMeta(
            String mid, AirportManager airportManager, MailFlightData flight,
            Option<String> lang, Option<String> yaDomain) {
        DateTimeZone departureTz;

        if (flight.getDepartureTz().isDefined()) {
            departureTz = flight.getDepartureTz().get();

        } else {
            departureTz = chooseTimezoneForPoint(
                    airportManager, flight.getDeparturePoint(), flight.getDepartureDateTime());
        }
        Option<DateTimeZone> arrivalTz = findTimezoneForPoint(airportManager, flight.getArrivalPoint());

        if (arrivalTz.isEmpty() && flight.getPlannedArrivalDateTime().isDefined()) {
            arrivalTz = Option.some(flight.getPlannedArrivalDateTime().get().getZone());
        }
        if (arrivalTz.isEmpty() && flight.getArrivalDateTime().isDefined()) {
            arrivalTz = Option.some(chooseTimezoneForPoint(
                    airportManager, flight.getArrivalPoint(), flight.getArrivalDateTime().get()));
        }

        DateTime departureDateTime = flight.getDepartureDateTime().toDateTime(departureTz);
        Option<DateTime> arrivalDateTime;

        if (!flight.isSegmented()
                && flight.getPlannedArrivalDateTime().isDefined()
                && flight.getPlannedDepartureDateTime().isDefined()) {
            Duration shift = new Duration(flight.getPlannedDepartureDateTime().get(), departureDateTime);
            arrivalDateTime = Option.some(flight.getPlannedArrivalDateTime().get().toLocalDateTime()
                    .toDateTime(arrivalTz.get()).plus(shift));

        } else if (flight.getArrivalDateTime().isDefined()) {
            arrivalDateTime = Option.some(flight.getArrivalDateTime(arrivalTz.get()).get());

        } else if (flight.getPlannedArrivalDateTime().isDefined()) {
            arrivalDateTime = flight.getPlannedArrivalDateTime(arrivalTz.get());

        } else {
            arrivalDateTime = Option.none();
        }

        if (arrivalDateTime.isDefined() && !departureDateTime.isBefore(arrivalDateTime.get())) {
            DateTime prevArrivalDateTime = arrivalDateTime.get();

            long deltaMs = departureDateTime.getMillis() - arrivalDateTime.get().getMillis()
                    + 30 * DateTimeConstants.MILLIS_PER_MINUTE;

            int tzOffsetHours = (int) Math.floorDiv(
                    arrivalTz.get().getOffset(arrivalDateTime.get()) - deltaMs, DateTimeConstants.MILLIS_PER_HOUR);

            Option<DateTimeZone> fixedArrivalTz;

            try {
                DateTimeZone fixedTz = DateTimeZone.forOffsetHours(tzOffsetHours);
                fixedArrivalTz = Option.some(DateTimeZone.forID(fixedTz.getID()));

            } catch (IllegalArgumentException e) {
                fixedArrivalTz = Option.none();
            }

            if (fixedArrivalTz.isDefined()) {
                arrivalDateTime = Option.some(arrivalDateTime.get().toLocalDateTime().toDateTime(fixedArrivalTz.get()));

                logger.warn("Guessed departure time {} is not before arrival time {}. Arrival time changed to {}",
                        departureDateTime.toString("yyyy-MM-dd HH:mm:ss (ZZZ)"),
                        prevArrivalDateTime.toString("yyyy-MM-dd HH:mm:ss (ZZZ)"),
                        arrivalDateTime.get().toString("yyyy-MM-dd HH:mm:ss (ZZZ)"));
            } else {
                logger.warn("Guessed departure time {} is not before arrival time {}. Can not change arrival time.",
                        departureDateTime.toString("yyyy-MM-dd HH:mm:ss (ZZZ)"),
                        prevArrivalDateTime.toString("yyyy-MM-dd HH:mm:ss (ZZZ)"));
            }
        }

        String flightNumber = flight.getFlightNumber();
        String normFlightNumber = normalizeFlightNumber(flightNumber);
        if (!flightNumber.equals(normFlightNumber)) {
            logger.warn("flight number from MAIL normalized: '{}' -> '{}'", flightNumber, normFlightNumber);
        }

        FlightPoint departurePoint = findActualFlightPointInfo(airportManager, flight.getDeparturePoint());
        FlightPoint arrivalPoint = findActualFlightPointInfo(airportManager, flight.getArrivalPoint());

        return new FlightEventMeta(mid, flightNumber, flight.getAirline(),
                flight.getPlannedDepartureDateTime().map(DateTime::toLocalDateTime),
                departurePoint.getCity(), departurePoint.getAirport(),
                departureDateTime.toLocalDateTime(), departureDateTime.getZone(),
                arrivalPoint.getCity(), arrivalPoint.getAirport(),
                arrivalDateTime.map(DateTime::toLocalDateTime), arrivalDateTime.map(DateTime::getZone),
                flight.getSource(), flight.getCheckInLink(), flight.getAeroexpressLink(),
                flight.getLastSegmentFlightNumber(), flight.getLastSegmentDepartureDateTime(),
                flight.getLastSegmentSource(), flight.getDirection(), lang, yaDomain);
    }

    private static Option<DateTimeZone> findTimezoneForPoint(AirportManager airportManager, FlightPoint point) {
        return airportManager.findTimezoneByAirportCodesOrCityGeoId(
                point.getIataCode(), point.getIcaoCode(), point.getGeoId());
    }

    private static DateTimeZone chooseTimezoneForPoint(
            AirportManager airportManager, FlightPoint point, LocalDateTime dateTime) {
        Option<DateTimeZone> tz = findTimezoneForPoint(airportManager, point);

        if (tz.isEmpty()) {
            return airportManager.chooseTimezoneForCityAndAirportName(
                    point.getCityName(), point.getAirportName(), dateTime);
        } else {
            return tz.get();
        }
    }

    private static FlightPoint findActualFlightPointInfo(AirportManager airportManager, FlightPoint point) {
        Option<Airport> found = airportManager.findAirportByCodes(point.getIataCode(), point.getIcaoCode());

        if (found.isDefined()) {
            FlightCity city = new FlightCity(
                    point.getCityName(),
                    found.get().getCityGeoId().map(Cf.Long.toIntegerF()).orElse(point.getCity().getGeoId()));

            FlightItem airport = point.getAirport().get();

            airport = new FlightItem(
                    airport.getName(),
                    airport.getRaspId(),
                    found.get().getIcao().orElse(airport.getIcaoCode()),
                    found.get().getIata().orElse(airport.getIataCode()),
                    airport.getSirenaCode());

            return new FlightPoint(city, Option.some(airport));
        }
        return point;
    }

    public static String normalizeFlightNumber(String flightNumber) {
        return FLIGHT_NUMBER_NORMALIZE_PATTERN.replaceAll(flightNumber, Function.<String, String>constF(""))
                .toUpperCase();
    }

}
