package ru.yandex.reminders.api.flight;

import lombok.val;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDateTime;
import org.springframework.beans.factory.annotation.Autowired;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.commune.a3.action.Action;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.result.json.JsonWriterActionResult;
import ru.yandex.commune.json.write.JsonWriter;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.thread.WithTlTimeoutInMillis;
import ru.yandex.reminders.api.StatusResult;
import ru.yandex.reminders.api.a3.bind.BindJson;
import ru.yandex.reminders.log.TskvFields;
import ru.yandex.reminders.log.TskvLogDataSource;
import ru.yandex.reminders.log.TskvName;
import ru.yandex.reminders.logic.event.SpecialClientIds;
import ru.yandex.reminders.logic.flight.FlightEventMeta;
import ru.yandex.reminders.logic.flight.FlightReminderManager;
import ru.yandex.reminders.logic.flight.airport.AirportManager;
import ru.yandex.reminders.logic.reminder.PhoneNumber;
import ru.yandex.reminders.logic.reminder.SendDailyStat;
import ru.yandex.reminders.logic.update.ActionInfo;
import ru.yandex.reminders.util.HostnameUtils;

@ActionContainer
public class FlightRemindersActions implements TskvLogDataSource {

    private static final Logger logger = LoggerFactory.getLogger(FlightRemindersActions.class);

    @Autowired
    private FlightReminderManager flightReminderManager;
    @Autowired
    private AirportManager airportManager;

    private static ActionInfo actionInfo() {
        return new ActionInfo(Instant.now(), RequestIdStack.current().get(), HostnameUtils.getLocalhostId());
    }

    @Action
    @WithTlTimeoutInMillis(10000)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_MS)
    public MailFlightRemindersInfo getFlightRemindersInfo(@BindJson MailFlightRemindersData data) {
        val flightEventMetas = FlightDataConverter.toFlightEventMetas(data, airportManager);
        return MailFlightRemindersInfo.cons(
                flightReminderManager.getFlightRemindersInfo(
                        data.getUid(), data.getTicketId(), flightEventMetas, actionInfo()));
    }

    @Action
    @WithTlTimeoutInMillis(10000)
    public MailFlightRemindersInfo createOrUpdateFlightReminders(@BindJson MailFlightRemindersData data) {
        data.validate();
        val flightEventMetas = FlightDataConverter.toFlightEventMetas(data, airportManager);
        // for checking https://jira.yandex-team.ru/browse/DARIA-28542
        Validate.unique(
                flightEventMetas.map(FlightEventMeta.getDepartureTsF()),
                "Departure time should be unique among flights");
        return MailFlightRemindersInfo.cons(
                flightReminderManager.createOrUpdateFlightReminders(
                        data.getUid(), data.getTicketId(), data.getReminderData(), flightEventMetas, actionInfo()
                )
        );
    }

    @Action
    @WithTlTimeoutInMillis(10000)
    public StatusResult deleteFlightReminders(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("ticketId") @TskvName(TskvFields.ID) String ticketId) {
        flightReminderManager.deleteFlightReminders(uid, ticketId, actionInfo());
        return StatusResult.OK;
    }

    @Action
    @WithTlTimeoutInMillis(10000)
    public StatusResult deleteSmsFlightReminders(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("ticketId") @TskvName(TskvFields.ID) String ticketId) {
        flightReminderManager.deleteSmsFlightReminders(uid, ticketId);
        return StatusResult.OK;
    }

    @Action
    @WithTlTimeoutInMillis(10000)
    public StatusResult sendPromiseSms(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("phone") Option<String> phone,
            @RequestParam("offset") int offset,
            @RequestParam("lang") Option<String> lang) {
        flightReminderManager.sendPromiseSms(
                uid, phone.map(PhoneNumber.phoneF()), Duration.standardMinutes(offset), lang);
        return StatusResult.OK;
    }

    @Action
    @WithTlTimeoutInMillis(10000)
    public StatusResult disableFlightReminders(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("disabled") Option<Boolean> disabled) {
        flightReminderManager.disableFlightReminders(uid, disabled.getOrElse(true));
        return StatusResult.OK;
    }

    @Action
    public JsonWriterActionResult getSmsFlightRemindersSendStats(
            @RequestParam("allHistory") Option<Boolean> all) {
        val stats = flightReminderManager.getSmsFlightRemindersSendStats(all.getOrElse(false));

        return (jw, additionalAttributes) -> {
            jw.writeArrayStart();

            for (SendDailyStat stat : stats) {
                jw.writeObjectStart();
                jw.writeStringField("date", stat.getDate().toString("yyyyMMdd"));
                jw.writeStringField("name", "scheduled");
                jw.writeNumberField("data", stat.getTotal());
                jw.writeObjectEnd();

                jw.writeObjectStart();
                jw.writeStringField("date", stat.getDate().toString("yyyyMMdd"));
                jw.writeStringField("name", "sent");
                jw.writeNumberField("data", stat.getSent());
                jw.writeObjectEnd();
            }
            jw.writeArrayEnd();
        };
    }

    @Action
    @WithTlTimeoutInMillis(10000)
    public StatusResult flightTimeChanged(
            @RequestParam("flight") String flightNumber,
            @RequestParam("geoId") int geoId,
            @RequestParam("tz") DateTimeZone tz,
            @RequestParam("plannedDateTime") LocalDateTime plannedDateTime,
            @RequestParam("actualDateTime") LocalDateTime actualDateTime) {
        Validate.notBlank(flightNumber, "flight should not be blank");
        val normFlightNumber = FlightDataConverter.normalizeFlightNumber(flightNumber);
        Validate.notBlank(normFlightNumber, "flight should not be blank after normalization");
        if (!flightNumber.equals(normFlightNumber)) {
            logger.info("flight number from RASP normalized: '{}' -> '{}'", flightNumber, normFlightNumber);
        }
        Validate.gt(geoId, 0, "geoId should be greater than 0");
        flightReminderManager.flightTimeChanged(
                normFlightNumber, geoId, tz, plannedDateTime, actualDateTime, actionInfo());
        return StatusResult.OK;
    }

    @Override
    public Tuple2List<String, String> tskvLogData() {
        return Tuple2List.fromPairs(TskvFields.CID, SpecialClientIds.FLIGHT);
    }
}
