package ru.yandex.reminders.api.flight;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.LocalDateTime;
import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderMembersToBind;
import ru.yandex.misc.lang.Validate;
import ru.yandex.reminders.logic.flight.FlightCity;
import ru.yandex.reminders.logic.flight.FlightItem;
import ru.yandex.reminders.logic.flight.FlightSource;

@Bendable
@BenderMembersToBind(MembersToBind.ALL_FIELDS)
public class MailFlightData {
    private String flightNumber;
    private Option<FlightItem> airline;

    private FlightCity departureCity;
    private Option<FlightItem> departureAirport;

    private FlightCity arrivalCity;
    private Option<FlightItem> arrivalAirport;

    private LocalDateTime departureDateTime;
    private Option<LocalDateTime> arrivalDateTime;

    private Option<DateTime> plannedDepartureDateTime;
    private Option<DateTime> plannedArrivalDateTime;

    private Option<DateTimeZone> departureTz;
    private Option<FlightSource> source;

    private Option<String> checkInLink;
    private Option<String> aeroexpressLink;

    private Option<String> lastSegmentFlightNumber;
    private Option<LocalDateTime> lastSegmentDepartureDateTime;
    private Option<FlightSource> lastSegmentSource;

    private Option<String> direction;

    private MailFlightData(
            String flightNumber, Option<FlightItem> airline,
            FlightCity departureCity, Option<FlightItem> departureAirport,
            FlightCity arrivalCity, Option<FlightItem> arrivalAirport,
            LocalDateTime departureDateTime, Option<LocalDateTime> arrivalDateTime,
            Option<DateTime> plannedDepartureDateTime, Option<DateTime> plannedArrivalDateTime,
            Option<DateTimeZone> departureTz, Option<FlightSource> source, Option<String> checkInLink,
            Option<String> aeroexpressLink, Option<String> lastSegmentFlightNumber,
            Option<LocalDateTime> lastSegmentDepartureDateTime,
            Option<FlightSource> lastSegmentSource, Option<String> direction) {
        this.flightNumber = flightNumber;
        this.airline = airline;
        this.departureCity = departureCity;
        this.departureAirport = departureAirport;
        this.arrivalCity = arrivalCity;
        this.arrivalAirport = arrivalAirport;
        this.departureDateTime = departureDateTime;
        this.arrivalDateTime = arrivalDateTime;
        this.plannedDepartureDateTime = plannedDepartureDateTime;
        this.plannedArrivalDateTime = plannedArrivalDateTime;
        this.departureTz = departureTz;
        this.source = source;
        this.checkInLink = checkInLink;
        this.aeroexpressLink = aeroexpressLink;
        this.lastSegmentFlightNumber = lastSegmentFlightNumber;
        this.lastSegmentDepartureDateTime = lastSegmentDepartureDateTime;
        this.lastSegmentSource = lastSegmentSource;
        this.direction = direction;
    }

    public static Builder builder() {
        return new Builder();
    }

    public void validate() {
        Validate.notBlank(flightNumber, "flightNumber should not be blank");
        Validate.notBlank(departureCity.getName(), "departureCity should not be blank");
        Validate.notNull(departureDateTime, "departureDateTime should not be null");
        Validate.notBlank(arrivalCity.getName(), "arrivalCity should not be blank");

        hackSource();
        if (source.isDefined()) {
            switch (source.get()) {
                case IEX:
                    break;
                case RASP:
                    Validate.some(plannedDepartureDateTime, "plannedDepartureDateTime should be specified");
                    Validate.some(departureTz, "departureTz should be specified");
                    Validate.some(departureCity.getGeoId(), "departureCity.geoId should be specified");
                    break;
                default:
                    Validate.fail("Unsupported source=" + source.get());
            }
        }
    }

    public String getFlightNumber() {
        return flightNumber;
    }

    public Option<FlightItem> getAirline() {
        return airline;
    }

    public FlightPoint getDeparturePoint() {
        return new FlightPoint(getDepartureCity(), getDepartureAirport());
    }

    public FlightCity getDepartureCity() {
        return departureCity;
    }

    public Option<FlightItem> getDepartureAirport() {
        return departureAirport;
    }

    public FlightPoint getArrivalPoint() {
        return new FlightPoint(getArrivalCity(), getArrivalAirport());
    }

    public FlightCity getArrivalCity() {
        return arrivalCity;
    }

    public Option<FlightItem> getArrivalAirport() {
        return arrivalAirport;
    }

    public LocalDateTime getDepartureDateTime() {
        return departureDateTime;
    }

    public DateTime getDepartureDateTime(DateTimeZone tz) {
        return departureDateTime.toDateTime(tz);
    }

    public Option<LocalDateTime> getArrivalDateTime() {
        return arrivalDateTime;
    }

    public Option<DateTime> getArrivalDateTime(DateTimeZone tz) {
        return arrivalDateTime.map(dt -> dt.toDateTime(tz));
    }

    public Option<DateTime> getPlannedDepartureDateTime() {
        return plannedDepartureDateTime;
    }

    public Option<DateTime> getPlannedArrivalDateTime() {
        return plannedArrivalDateTime;
    }

    public Option<DateTime> getPlannedArrivalDateTime(DateTimeZone tz) {
        return plannedArrivalDateTime.map(dt -> dt.toLocalDateTime().toDateTime(tz));
    }

    public Option<DateTimeZone> getDepartureTz() {
        return departureTz;
    }

    public Option<FlightSource> getSource() {
        return source;
    }

    public Option<String> getCheckInLink() {
        return checkInLink;
    }

    public Option<String> getAeroexpressLink() {
        return aeroexpressLink;
    }

    public Option<String> getLastSegmentFlightNumber() {
        return lastSegmentFlightNumber;
    }

    public Option<LocalDateTime> getLastSegmentDepartureDateTime() {
        return lastSegmentDepartureDateTime;
    }

    public Option<FlightSource> getLastSegmentSource() {
        return lastSegmentSource;
    }

    public Option<String> getDirection() {
        return direction;
    }

    public boolean isSegmented() {
        return lastSegmentDepartureDateTime.isDefined()
                || lastSegmentSource.isDefined()
                || lastSegmentFlightNumber.isDefined();
    }

    // tmp hack for integration with Pochta
    public void hackSource() {
        if (source.isSome(FlightSource.RASP)
                && (plannedDepartureDateTime.isEmpty() || departureTz.isEmpty() || departureCity.getGeoId().isEmpty())) {
            source = Option.none();
            lastSegmentSource = Option.none();
        }
    }

    public static class Builder {
        private final MailFlightData data = new MailFlightData(
                null, Option.none(), null, Option.none(), null, Option.none(), null,
                Option.none(), Option.none(), Option.none(), Option.none(), Option.none(), Option.none(),
                Option.none(), Option.none(), Option.none(), Option.none(), Option.none());


        public Builder setFlightNumber(String flightNumber) {
            data.flightNumber = flightNumber;
            return this;
        }

        public Builder setAirline(FlightItem airline) {
            data.airline = Option.some(airline);
            return this;
        }

        public Builder setDepartureCity(FlightCity departureCity) {
            data.departureCity = departureCity;
            return this;
        }

        public Builder setDepartureCity(String departureCity) {
            return setDepartureCity(new FlightCity(departureCity));
        }

        public Builder setDepartureAirport(FlightItem departureAirport) {
            data.departureAirport = Option.some(departureAirport);
            return this;
        }

        public Builder setDepartureDateTime(LocalDateTime departureDateTime) {
            data.departureDateTime = departureDateTime;
            return this;
        }

        public Builder setPlannedDepartureDateTime(DateTime plannedDepartureDateTime) {
            data.plannedDepartureDateTime = Option.some(plannedDepartureDateTime);
            return this;
        }

        public Builder setDepartureTz(DateTimeZone departureTz) {
            data.departureTz = Option.some(departureTz);
            return this;
        }

        public Builder setSource(FlightSource source) {
            data.source = Option.some(source);
            return this;
        }

        public Builder setArrivalCity(FlightCity arrivalCity) {
            data.arrivalCity = arrivalCity;
            return this;
        }

        public Builder setArrivalCity(String arrivalCity) {
            return setArrivalCity(new FlightCity(arrivalCity));
        }

        public Builder setArrivalAirport(FlightItem arrivalAirport) {
            data.arrivalAirport = Option.some(arrivalAirport);
            return this;
        }

        public Builder setArrivalDateTime(LocalDateTime dateTime) {
            data.arrivalDateTime = Option.some(dateTime);
            return this;
        }

        public Builder setPlannedArrivalDateTime(DateTime plannedDepartureDateTime) {
            data.plannedArrivalDateTime = Option.some(plannedDepartureDateTime);
            return this;
        }

        public Builder setCheckInLink(String checkInLink) {
            data.checkInLink = Option.some(checkInLink);
            return this;
        }

        public Builder setAeroexpressLink(String aeroexpressLink) {
            data.aeroexpressLink = Option.some(aeroexpressLink);
            return this;
        }

        public Builder setLastSegmentFlightNumber(String lastSegmentFlightNumber) {
            data.lastSegmentFlightNumber = Option.some(lastSegmentFlightNumber);
            return this;
        }

        public Builder setLastSegmentDepartureDateTime(LocalDateTime lastSegmentDepartureDateTime) {
            data.lastSegmentDepartureDateTime = Option.some(lastSegmentDepartureDateTime);
            return this;
        }

        public Builder setLastSegmentSource(FlightSource lastSegmentSource) {
            data.lastSegmentSource = Option.some(lastSegmentSource);
            return this;
        }

        public Builder setDirection(String direction) {
            data.direction = Option.some(direction);
            return this;
        }

        public MailFlightData build() {
            return new MailFlightData(
                    Option.notNull(data.getFlightNumber()).getOrThrow("flightNumber"),
                    data.getAirline(),
                    Option.notNull(data.getDepartureCity()).getOrThrow("departureCity"), data.getDepartureAirport(),
                    Option.notNull(data.getArrivalCity()).getOrThrow("arrivalCity"), data.getArrivalAirport(),
                    Option.notNull(data.getDepartureDateTime()).getOrThrow("departureDateTime"),
                    data.getArrivalDateTime(),
                    data.getPlannedDepartureDateTime(), data.getPlannedArrivalDateTime(),
                    data.getDepartureTz(), data.getSource(),
                    data.getCheckInLink(), data.getAeroexpressLink(),
                    data.getLastSegmentFlightNumber(), data.getLastSegmentDepartureDateTime(),
                    data.getLastSegmentSource(), data.getDirection());
        }

    }
}
