package ru.yandex.reminders.api.reminder;

import java.net.URI;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import lombok.val;
import one.util.streamex.StreamEx;
import org.joda.time.DateTime;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderFlatten;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.lang.Validate;
import ru.yandex.reminders.logic.reminder.Channel;
import ru.yandex.reminders.logic.reminder.Reminder;

@BenderBindAllFields
public class ChannelsData {
    private final Option<Sms> sms;
    private final Option<Mail> email;
    private final Option<Panel> panel;
    private final Option<Callback> callback;
    private final Option<Sup> sup;

    public ChannelsData(
            Option<Sms> sms, Option<Mail> email,
            Option<Panel> panel, Option<Callback> callback, Option<Sup> sup)
    {
        this.sms = sms;
        this.email = email;
        this.panel = panel;
        this.callback = callback;
        this.sup = sup;
    }

    public ChannelsData(
            Optional<Sms> sms, Optional<Mail> email,
            Optional<Panel> panel, Optional<Callback> callback, Optional<Sup> sup) {
        this(Option.wrap(sms), Option.wrap(email), Option.wrap(panel), Option.wrap(callback), Option.wrap(sup));
    }

    public void validate() {
        sms.forEach(Sms::validate);
        email.forEach(Mail::validate);
        panel.forEach(Panel::validate);
        callback.forEach(Callback::validate);
        sup.forEach(Sup::validate);
    }

    public static ChannelsData create(Collection<Reminder> reminders) {
        val channelMap = StreamEx.of(reminders).groupingBy(Reminder.getChannelF());

        return new ChannelsData(
                create(channelMap, Channel.SMS, ChannelsData.Sms.fromReminderF()),
                create(channelMap, Channel.EMAIL, ChannelsData.Mail.fromReminderF()),
                create(channelMap, Channel.PANEL, ChannelsData.Panel.fromReminderF()),
                create(channelMap, Channel.CALLBACK, ChannelsData.Callback.fromReminderF()),
                create(channelMap, Channel.SUP, ChannelsData.Sup.fromReminderF())
        );
    }

    private static <T> Optional<T> create(Map<Channel, List<Reminder>> channelMap, Channel key, Function<? super Reminder, T> channelConstructor) {
        return channelMap.getOrDefault(key, Collections.emptyList()).stream()
                .map(channelConstructor)
                .findFirst();
    }

    public boolean isEmpty() {
        return !sms.isPresent() && !email.isPresent() && !panel.isPresent() && !callback.isPresent() && !sup.isPresent();
    }

    public Option<Sms> getSms() {
        return sms;
    }

    public Option<Mail> getEmail() {
        return email;
    }

    public Option<Panel> getPanel() {
        return panel;
    }

    public Option<Callback> getCallback() {
        return callback;
    }

    public Option<Sup> getSup() {
        return sup;
    }

    @BenderBindAllFields
    public static class Sms {
        private final Option<String> text;

        public Sms(Option<String> text) {
            this.text = text;
        }

        public Reminder toReminder(DateTime reminderDate) {
            return Reminder.sms(reminderDate, Option.empty(), Option.empty(), Option.empty(), text);
        }

        public void validate() {
            text.forEach(txt -> LenLimits.SMS.checkParam("sms", txt));
        }

        public static Function<Reminder, Sms> fromReminderF() {
            return r -> {
                Validate.isTrue(r.isSms());
                return new Sms(r.getText());
            };
        }

        public static Function<Sms, Reminder> toReminderF(final DateTime reminderDate) {
            return s -> s.toReminder(reminderDate);
        }
    }

    @BenderBindAllFields
    public static class Mail {
        private final Option<String> from; // CAL-6586
        private final Option<String> subject;
        private final Option<String> bodyHtml;
        private final Option<String> bodyText;

        public Mail(Option<String> from, Option<String> subject, Option<String> bodyHtml, Option<String> bodyText) {
            this.from = from;
            this.subject = subject;
            this.bodyHtml = bodyHtml;
            this.bodyText = bodyText;
        }

        public Reminder toReminder(DateTime reminderDate) {
            return Reminder.email(
                    reminderDate, Option.empty(), Option.empty(),
                    from, Option.empty(), subject, bodyText, bodyHtml);
        }

        public void validate() {
            LenLimits.EMAIL.checkParam("email", from.plus(subject).plus(bodyHtml).plus(bodyText).mkString(""));
        }

        public static Function<Reminder, Mail> fromReminderF() {
            return r -> {
                Validate.isTrue(r.isEmail());
                return new Mail(r.getFrom(), r.getSubject(), r.getBodyHtml(), r.getBodyText());
            };
        }

        public static Function<Mail, Reminder> toReminderF(final DateTime reminderDate) {
            return m -> m.toReminder(reminderDate);
        }
    }

    @BenderBindAllFields
    public static class Panel {
        private final Option<String> url;
        private final Option<String> subject;
        private final Option<String> message;

        public Panel(Option<String> url, Option<String> subject, Option<String> message) {
            this.url = url;
            this.subject = subject;
            this.message = message;
        }

        public Reminder toReminder(DateTime reminderDate) {
            return Reminder.panel(reminderDate, url, subject, message);
        }

        public void validate() {
        }

        public static Function<Reminder, Panel> fromReminderF() {
            return r -> {
                Validate.isTrue(r.isPanel());
                return new Panel(r.getUrl(), r.getSubject(), r.getMessage());
            };
        }

        public static Function<Panel, Reminder> toReminderF(final DateTime reminderDate) {
            return p -> p.toReminder(reminderDate);
        }
    }

    @BenderBindAllFields
    public static class Callback {
        private final String url;

        public Callback(String url) {
            this.url = url;
        }

        public Reminder toReminder(DateTime reminderDate) {
            return Reminder.callback(reminderDate, url);
        }

        public void validate() {
            try {
                Validate.isTrue(URI.create(url).isAbsolute(), "url should be absolute");

            } catch (RuntimeException e) {
                Validate.fail("invalid url, e=" + e.getMessage());
            }
        }

        public static Function<Reminder, Callback> fromReminderF() {
            return r -> {
                Validate.isTrue(r.isCallback());
                return new Callback(r.getUrl().get());
            };
        }

        public static Function<Callback, Reminder> toReminderF(final DateTime reminderDate) {
            return c -> c.toReminder(reminderDate);
        }
    }

    @BenderBindAllFields
    public static class Sup {
        private final Option<String> title;
        private final Option<String> body;
        private final Option<String> uri;
        @BenderFlatten
        private final SupExtraFields extraFields;

        @Bendable
        public static class SupExtraFields {
            @BenderPart(name="device_id", strictName = true)
            private final Optional<String> deviceId;
            private final Optional<String> uuid;
            @BenderPart(name="client_id", strictName = true)
            private final Optional<String> clientId;

            public SupExtraFields() {
                this.deviceId = Optional.empty();
                this.uuid = Optional.empty();
                this.clientId = Optional.empty();
            }

            public SupExtraFields(Optional<String> deviceId, Optional<String> uuid, Optional<String> clientId) {
                this.deviceId = deviceId;
                this.uuid = uuid;
                this.clientId = clientId;
            }

            public Optional<String> getDeviceId() {
                return deviceId;
            }
        }

        public Sup(Option<String> title, Option<String> body, Option<String> uri, SupExtraFields extraFields) {
            this.title = title;
            this.body = body;
            this.uri = uri;
            this.extraFields = extraFields;
        }

        public void validate() {
            uri.map(URI::create);
        }

        public static Function<Reminder, Sup> fromReminderF() {
            return r -> {
                Validate.isTrue(r.isSup());
                return new Sup(r.getSubject(), r.getMessage(), r.getUrl(), r.getSupExtraFields());
            };
        }

        public static Function<Sup, Reminder> toReminderF(DateTime reminderDate) {
            return c -> Reminder.sup(reminderDate, c.title, c.body, c.uri, c.extraFields);
        }
    }
}
