package ru.yandex.reminders.api.reminder;

import java.util.Optional;
import java.util.function.Supplier;
import java.util.stream.Stream;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.StreamEx;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.a3.action.A3Exception;
import ru.yandex.commune.a3.action.Action;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.IllegalParameterException;
import ru.yandex.commune.a3.action.parameter.ValidateParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.PathParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.result.error.CommonErrorNames;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.thread.WithTlTimeoutInMillis;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.misc.time.MoscowTime;
import ru.yandex.reminders.api.ErrorResult;
import ru.yandex.reminders.api.StatusResult;
import ru.yandex.reminders.api.a3.bind.BindJson;
import ru.yandex.reminders.logic.callmeback.CallmebackManager;
import ru.yandex.reminders.logic.event.EventData;
import ru.yandex.reminders.logic.event.EventId;
import ru.yandex.reminders.logic.event.EventManager;
import ru.yandex.reminders.logic.event.EventsFilter;
import ru.yandex.reminders.logic.event.IterationKey;
import ru.yandex.reminders.logic.event.Order;
import ru.yandex.reminders.logic.event.SpecialClientIds;
import ru.yandex.reminders.logic.reminder.Origin;
import ru.yandex.reminders.logic.reminder.Reminder;
import ru.yandex.reminders.logic.sup.SupClientSettingsRegistry;
import ru.yandex.reminders.logic.update.ActionInfo;
import ru.yandex.reminders.util.HostnameUtils;

@ActionContainer
@Slf4j
public class ReminderActions {
    @Autowired
    private EventManager eventManager;
    @Autowired
    private SupClientSettingsRegistry supClientSettingsRegistry;
    @Autowired
    private CallmebackManager callmebackManager;
    private static final String[] MONGO_CLIENTS = {SpecialClientIds.WMI, SpecialClientIds.CALENDAR};

    private static ActionInfo actionInfo() {
        return new ActionInfo(Instant.now(), RequestIdStack.current().get(), HostnameUtils.getLocalhostId());
    }

    private Optional<ReminderInfo> getMongoSingleEvent(PassportUid uid, String clientId, String externalId) {
        val filter = EventsFilter.byExternalId(externalId);
        val event = eventManager.findEvents(uid, clientId, filter).singleO();
        return event.toOptional().map(ReminderDataConverter::convertToReminderInfo);
    }

    private Optional<ReminderInfo> getCallmebackSingleEvent(EventId id) {
        if (isClientMongoOnly(id.getCid())) {
            return Optional.empty();
        }
        return callmebackManager.findEvent(id);
    }

    private static boolean isClientMongoOnly(String clientId) {
        return Stream.of(MONGO_CLIENTS).anyMatch(client -> client.equals(clientId));
    }

    @Action
    @WithTlTimeoutInMillis(3000)
    @Path(methods = HttpMethod.GET, value = "/v1/{uid}/reminders/{cid}/")
    public Object findReminder(
            @PathParam("uid") PassportUid uid, @PathParam("cid") String clientId,
            @RequestParam("id") Option<String> externalId, @RequestParam("interval") Option<String> intervalStr,
            @RequestParam("iterationKey") Option<IterationKey> iterationKey,
            @RequestParam("newFirst") Option<Boolean> newFirst,
            @RequestParam("soonestFirst") Option<Boolean> soonestFirst,
            @RequestParam("count") Option<Integer> count) {
        if (externalId.isPresent()) {
            Supplier<Optional<ReminderInfo>> callmebackSupplier = () -> getCallmebackSingleEvent(new EventId(uid, clientId, externalId.get()));
            Supplier<Optional<ReminderInfo>> mongoSupplier = () -> getMongoSingleEvent(uid, clientId, externalId.get());

            return first(new SupplierDesc("callmeback", callmebackSupplier), new SupplierDesc("mongo", mongoSupplier))
                    .map(a -> (Object) a)
                    .orElse(new ErrorResult(CommonErrorNames.NOT_FOUND, "Reminder not found by id " + externalId.get(),
                            HttpStatus.SC_404_NOT_FOUND));
        } else {
            if (isClientMongoOnly(clientId)) {
                return getMongoEventList(uid, clientId, intervalStr.toOptional(), iterationKey.toOptional(),
                        newFirst.toOptional(), soonestFirst.toOptional(), count.toOptional());
            }
            return getCallmebackEventList(uid, clientId, intervalStr.toOptional(), iterationKey.toOptional(),
                    newFirst.toOptional(), soonestFirst.toOptional(), count.toOptional());
        }
    }

    private Object getCallmebackEventList(PassportUid uid, String clientId, Optional<String> intervalStr,
                                                 Optional<IterationKey> iterationKey,
                                                 Optional<Boolean> newFirst,
                                                 Optional<Boolean> soonestFirst,
                                                 Optional<Integer> count) {
        if (Stream.of(intervalStr, iterationKey, newFirst, soonestFirst, count).anyMatch(Optional::isPresent)) {
            return new ErrorResult(CommonErrorNames.VALIDATE, "GET interval query with additional parameters is only supported for the following clients: "
                    + String.join(",", MONGO_CLIENTS), HttpStatus.SC_403_FORBIDDEN);
        }

        return callmebackManager.listEvents(uid, clientId);
    }

    private RemindersInfo getMongoEventList(PassportUid uid, String clientId, Optional<String> intervalStr,
                                                   Optional<IterationKey> iterationKey,
                                                   Optional<Boolean> newFirst,
                                                   Optional<Boolean> soonestFirst,
                                                   Optional<Integer> count) {
        intervalStr.ifPresent(v -> ValidateParam.some("interval", ReminderDataConverter.parseIntervalO(v), "Parameter 'interval' value is malformed"));
        val interval = intervalStr.map(v -> ReminderDataConverter.parseIntervalO(v).get())
                .orElse(new InstantInterval(actionInfo().getNow(), MoscowTime.dateTime(2100, 1, 1, 0, 0)));

        val totalFilter = EventsFilter.any().andByReminderSendTs(interval);
        val filter = iterationKey.map(totalFilter::andWithPrevId).orElse(totalFilter).withOrder(Order.of(soonestFirst, newFirst));
        val filterWithLimits = count.map(v -> filter.withLimits(SqlLimits.first(v + 1))).orElse(filter);

        val found = eventManager.findEvents(uid, clientId, filterWithLimits);
        val events = count.map(found::take).orElse(found);
        val nextKey = Option.when(found.size() != events.size(), () -> IterationKey.of(events.last()));

        Optional<Long> totalCount = Optional.empty();

        if (count.isPresent() || iterationKey.isPresent()) {
            if (!iterationKey.isPresent() && !nextKey.isPresent()) {
                totalCount = Optional.of((long) events.size());
            } else {
                totalCount = Optional.of(eventManager.countEvents(uid, clientId, totalFilter));
            }
        }
        return new RemindersInfo(events.map(ReminderDataConverter::convertToReminderInfo), nextKey, Option.wrap(totalCount));
    }

    @Action
    @WithTlTimeoutInMillis(3000)
    @Path(methods = HttpMethod.POST, value = "/v1/{uid}/reminders/{cid}/")
    public ReminderIdInfo createReminder(
            @PathParam("uid") PassportUid uid, @PathParam("cid") String clientId,
            @RequestParam("senderName") Option<String> senderName,
            @RequestParam("source") Option<Source> source,
            @BindJson ReminderData data) {
        EventData eventData = ReminderDataConverter.convertToEventData(data, source, clientId);
        eventData = addAutoXivaSup(clientId, eventData);

        val eventId = new EventId(uid, clientId);

        if (isClientMongoOnly(clientId)) {
            eventManager.createOrUpdateEvent(eventId, eventData, senderName, actionInfo());
        } else {
            callmebackManager.createEvent(eventId, eventData, senderName.toOptional());
        }

        return new ReminderIdInfo(eventId.getExtId());
    }

    @Action
    @WithTlTimeoutInMillis(3000)
    @Path(methods = HttpMethod.PUT, value = "/v1/{uid}/reminders/{cid}/")
    public ReminderIdInfo createOrUpdateReminder(
            @PathParam("uid") PassportUid uid, @PathParam("cid") String clientId,
            @RequestParam("id") String externalId,
            @RequestParam("senderName") Option<String> senderName,
            @RequestParam("source") Option<Source> source,
            @BindJson ReminderData data) {
        ValidateParam.isTrue("id", StringUtils.isNotBlank(externalId), "Parameter 'id' should not be blank");
        EventData eventData = ReminderDataConverter.convertToEventData(data, source, clientId);
        eventData = addAutoXivaSup(clientId, eventData);
        val eventId = new EventId(uid, clientId, externalId);

        if (isClientMongoOnly(clientId)) {
            eventManager.createOrUpdateEvent(eventId, eventData, senderName, actionInfo());
        } else {
            callmebackManager.createOrUpdateEvent(eventId, eventData, senderName.toOptional());
            // Still remove that from mongo, if there is any
            eventManager.deleteEvents(uid, clientId, EventsFilter.byExternalId(externalId));
        }

        return new ReminderIdInfo(externalId);
    }

    @Action
    @WithTlTimeoutInMillis(3000)
    @Path(methods = HttpMethod.PUT, value = "/v1/{uid}/callmeback/{cid}/")
    public StatusResult transferReminder(
            @PathParam("uid") PassportUid uid, @PathParam("cid") String clientId,
            @RequestParam("id") String externalId) {
        ValidateParam.isTrue("id", StringUtils.isNotBlank(externalId), "Parameter 'id' should not be blank");
        val eventId = new EventId(uid, clientId, externalId);

        log.debug("Migrating event {} from reminders to callmeback...", eventId);

        val event = eventManager.findEvent(eventId);
        if (!event.isPresent()) {
            throw new A3Exception("transfer-fail", String.format("Failed to find event %s", eventId));
        }

        callmebackManager.createOrUpdateEvent(eventId, event.get().getEventData(), event.get().getSenderName().toOptional());
        // Still remove that from mongo, if there is any
        eventManager.deleteEvents(uid, clientId, EventsFilter.byExternalId(externalId));

        return StatusResult.OK;
    }

    @Action
    @WithTlTimeoutInMillis(3000)
    @Path(methods = HttpMethod.POST, value = "/v1/{uid}/reminders/{cid}/.put") // CAL-6588
    public ReminderIdInfo createOrUpdateReminderTemporary(
            @PathParam("uid") PassportUid uid, @PathParam("cid") String clientId,
            @RequestParam("id") String externalId,
            @RequestParam("senderName") Option<String> senderName,
            @RequestParam("source") Option<Source> source,
            @BindJson ReminderData data) {
        return createOrUpdateReminder(uid, clientId, externalId, senderName, source, data);
    }

    private EventData addAutoXivaSup(String clientId, EventData data) {
        val reminder = data.getReminders().find(Reminder::isSup);

        if (!reminder.isPresent()) return data;

        val settings = supClientSettingsRegistry.getO(clientId);

        if (!settings.isPresent()) throw new IllegalParameterException("sup", "Sup is not configured");

        if (!settings.get().getXivaPushToken().isPresent()) return data;

        if (!reminder.get().getUrl().isPresent()) return data;

        return data.withReminders(data.getReminders()
                .plus(Reminder.xiva(reminder.get().getSendDate(), Origin.AUTO, reminder.get().getUrl().get()))
                .sortedBy(Reminder::getSendDate));
    }

    @Action
    @WithTlTimeoutInMillis(3000)
    @Path(methods = HttpMethod.DELETE, value = "/v1/{uid}/reminders/{cid}/")
    public StatusResult deleteReminder(
            @PathParam("uid") PassportUid uid, @PathParam("cid") String clientId, @RequestParam("id") String externalId) {
        val eventId = new EventId(uid, clientId, externalId);
        if (!isClientMongoOnly(clientId)) {
            callmebackManager.removeEvent(eventId);
        }
        eventManager.deleteEvents(uid, clientId, EventsFilter.byExternalId(externalId));
        return StatusResult.OK;
    }

    @Action
    @WithTlTimeoutInMillis(3000)
    @Path(methods = HttpMethod.POST, value = "/v1/{uid}/reminders/{cid}/.delete") // CAL-6506
    public StatusResult deleteReminderTemporary(
            @PathParam("uid") PassportUid uid, @PathParam("cid") String clientId, @RequestParam("id") String externalId) {
        return deleteReminder(uid, clientId, externalId);
    }

    private static Optional<ReminderInfo> first(SupplierDesc... suppliers) {
        return StreamEx.of(suppliers)
                .map(SupplierDesc::getSupplier)
                .map(Supplier::get)
                .flatMap(StreamEx::of)
                .findFirst();
    }
}
