package ru.yandex.reminders.api.reminder;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.joda.time.Interval;
import org.joda.time.chrono.ISOChronology;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.a3.action.parameter.ValidateParam;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.reminders.logic.event.Event;
import ru.yandex.reminders.logic.event.EventData;
import ru.yandex.reminders.logic.event.SpecialClientIds;
import ru.yandex.reminders.logic.reminder.Channel;
import ru.yandex.reminders.logic.reminder.Reminder;

@Slf4j
public class ReminderDataConverter {
    public static ReminderInfo convertToReminderInfo(Event event) {
        validateEventData(event.getEventData());

        val channels = ChannelsData.create(event.getReminders());

        val id = event.getExternalId();
        val clientId = event.getClientId();

        val data = new ReminderData(
                event.getName(), event.getDescription(),
                Option.of(event.getReminders().map(Reminder.getSendDateF()).first()), channels, event.getData());

        return new ReminderInfo(id, clientId, data);
    }

    public static EventData convertToEventData(ReminderData data, Option<Source> source, String clientId) {
        if (!source.isPresent()) {
            source = Option.of(SpecialClientIds.hackSource(clientId));
            log.debug("source wasn't specified, auto-detected by clientId as {}", source.get());
        }

        val s = source.get();

        ValidateParam.isTrue("name", data.getName().exists(StringUtils.notBlankF()), "Field 'name' is required");
        LenLimits.SMS.checkParam("name", data.getName().get());

        val reminderDate = data.getReminderDate();
        ValidateParam.some("reminderDate", reminderDate, "Field 'reminderDate' is required");

        val eventDataValue = data.getData();
        if (eventDataValue.isPresent()) {
            LenLimits.JSON_DATA.checkParam("data", eventDataValue.get().toString());
        }
        val eventData = eventDataValue.filterByType(JsonObject.class).singleO();
        ValidateParam.sameSize("data", eventDataValue, eventData, "Field 'data' value expected to be an object");

        val channels = data.getChannels();

        ValidateParam.isFalse("channels", channels.isEmpty(),
                "Field 'channels' should contain at least one channel to remind");
        channels.validate();

        ListF<Reminder> reminders;

        boolean extSourceOrNezabudka = s == Source.EXTERNAL || SpecialClientIds.isNezabudka(clientId);
        if (extSourceOrNezabudka) { // CAL-6537, CAL-6572
            if (channels.getSms().isPresent()) {
                log.debug("ignoring sms channel cause source=EXTERNAL");
            }
            reminders = Cf.list();
        } else {
            reminders = channels.getSms().map(ChannelsData.Sms.toReminderF(reminderDate.get()));
        }

        reminders = reminders
                .plus(channels.getEmail().map(ChannelsData.Mail.toReminderF(reminderDate.get())))
                .plus(channels.getPanel().map(ChannelsData.Panel.toReminderF(reminderDate.get())))
                .plus(channels.getCallback().map(ChannelsData.Callback.toReminderF(reminderDate.get())))
                .plus(channels.getSup().map(ChannelsData.Sup.toReminderF(reminderDate.get())));

        if (extSourceOrNezabudka) { // CAL-6536
            reminders = reminders.map(r -> {
                switch (r.getChannel()) {
                    case SMS: {
                        Reminder result = r.withNoneSmsText();
                        if (result != r) {
                            log.debug("ignoring sms text cause source=EXTERNAL or cid=NEZABUDKA");
                        }
                        return result;
                    }
                    case EMAIL: {
                        Reminder result = r.withNoneEmailParts();
                        if (result != r) {
                            log.debug("ignoring email parts cause source=EXTERNAL or cid=NEZABUDKA");
                        }
                        return result;
                    }
                    default:
                        return r;
                }
            });
        }

        if (SpecialClientIds.HOTEL.equals(clientId)
                && !reminders.exists(Reminder.channelIsF(Channel.PANEL))
                && reminders.exists(Reminder.channelIsF(Channel.EMAIL))) {
            reminders = reminders.plus(Reminder.panel(reminderDate.get(), Option.none(), Option.none(), Option.none()));
        }

        reminders = reminders.sortedBy(Reminder::getSendDate);

        val result = new EventData(source, data.getName(), data.getDescription(), eventData, reminders);
        validateEventData(result);
        return result;
    }

    private static void validateEventData(EventData eventData) {
        Validate.unique(eventData.getReminders().map(Reminder.getChannelF()),
                "Event mustn't contain more than 1 reminder per each channel");
        Validate.notEmpty(eventData.getReminders(),
                "Event must contain at least 1 reminder");
        Validate.hasSize(1, eventData.getReminders().map(Reminder.getSendDateF()).unique(),
                "sendDate property should be the same for all event's reminders");
    }

    public static Option<InstantInterval> parseIntervalO(String value) {
        try {
            return Option.of(InstantInterval.valueOf(new Interval(value, ISOChronology.getInstanceUTC())));
        } catch (IllegalArgumentException e) {
            return Option.empty();
        }
    }
}
