package ru.yandex.reminders.api.tv;

import org.joda.time.DateTime;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.commune.a3.action.Action;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.PathParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestListParam;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.thread.WithTlTimeoutInMillis;
import ru.yandex.reminders.api.StatusResult;
import ru.yandex.reminders.api.a3.bind.BindJson;
import ru.yandex.reminders.log.TskvFields;
import ru.yandex.reminders.log.TskvLogDataSource;
import ru.yandex.reminders.logic.event.SpecialClientIds;
import ru.yandex.reminders.logic.tv.TvChannelId;
import ru.yandex.reminders.logic.tv.TvProgramId;
import ru.yandex.reminders.logic.tv.TvReminder;
import ru.yandex.reminders.logic.tv.TvReminderManager;
import ru.yandex.reminders.logic.update.ActionInfo;
import ru.yandex.reminders.util.HostnameUtils;

@ActionContainer
public class TvRemindersActions implements TskvLogDataSource {
    @Autowired
    private TvReminderManager tvReminderManager;

    @Action
    @WithTlTimeoutInMillis(10000)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_MS)
    @Path(methods = HttpMethod.GET, value = "/reminders/users/{uid}")
    public TvProgramsRemindersInfo getProgram(
            @PathParam("uid") PassportUid uid,
            @RequestListParam(value = "programIds", required = false) ListF<TvProgramId> programIds,
            @RequestListParam(value = "channelIds", required = false) ListF<TvChannelId> channelIds,
            @RequestListParam(value = "startTimes", required = false) ListF<DateTime> starts)
    {
        ListF<TvReminder> reminders = tvReminderManager.getTvProgramsReminders(uid, programIds, channelIds, starts);

        return new TvProgramsRemindersInfo(uid, reminders
                .toTuple2List(TvReminder.getChannelProgramIdF(), TvProgramEventData.fromReminderF())
                .groupBy1().mapEntries(TvProgramRemindersData.consF()));
    }

    @Action
    @WithTlTimeoutInMillis(10000)
    @Path(methods = HttpMethod.POST, value = "/reminders/users/{uid}")
    public StatusResult createOrUpdateProgram(
            @PathParam("uid") PassportUid uid, @BindJson TvProgramRemindersData data)
    {
        tvReminderManager.replaceTvProgramReminders(uid, data.getProgramId(), data.getReminders(), actionInfo());
        return StatusResult.OK;
    }

    @Action
    @WithTlTimeoutInMillis(10000)
    @Path(methods = HttpMethod.PATCH, value = "/reminders/users/{uid}")
    public StatusResult patchProgram(
            @PathParam("uid") PassportUid uid, @BindJson TvProgramRemindersData data)
    {
        tvReminderManager.addTvProgramReminders(uid, data.getProgramId(), data.getReminders(), actionInfo());
        return StatusResult.OK;
    }

    @Action
    @WithTlTimeoutInMillis(10000)
    @Path(methods = HttpMethod.DELETE, value = "/reminders/users/{uid}")
    public StatusResult deletePrograms(
            @PathParam("uid") PassportUid uid,
            @RequestListParam(value = "programIds", required = false) ListF<TvProgramId> programIds,
            @RequestListParam(value = "channelIds", required = false) ListF<TvChannelId> channelIds,
            @RequestListParam(value = "startTimes", required = false) ListF<DateTime> starts)
    {
        tvReminderManager.deleteTvProgramsReminders(uid, programIds, channelIds, starts, actionInfo());
        return StatusResult.OK;
    }

    @Action
    @WithTlTimeoutInMillis(10000)
    @Path(methods = HttpMethod.POST, value = "/reminders/users/{uid}/offset")
    public StatusResult changeOffset(
            @PathParam("uid") PassportUid uid, @BindJson TvProgramsRemindersOffsetData data)
    {
        tvReminderManager.changeTvProgramsRemindersOffset(uid, data.getOffsetMinutes(), actionInfo());
        return StatusResult.OK;
    }

    private static ActionInfo actionInfo() {
        return new ActionInfo(Instant.now(), RequestIdStack.current().get(), HostnameUtils.getLocalhostId());
    }

    @Override
    public Tuple2List<String, String> tskvLogData() {
        return Tuple2List.fromPairs(TskvFields.CID, SpecialClientIds.TV);
    }
}
