package ru.yandex.reminders.api.unistat;

import io.micrometer.core.instrument.Meter;
import io.micrometer.core.instrument.composite.CompositeMeterRegistry;
import io.micrometer.core.instrument.config.MeterFilter;
import io.micrometer.core.instrument.config.NamingConvention;
import io.micrometer.core.instrument.distribution.DistributionStatisticConfig;
import lombok.val;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import ru.yandex.kinopoisk.micrometer.yasm.YasmMeterRegistry;

import java.time.Duration;

@Configuration
public class UnistatConfiguration {
    private static final long[] histBucketsBounds = {
            getNanosByMillis(10),
            getNanosByMillis(15),
            getNanosByMillis(20),
            getNanosByMillis(25),
            getNanosByMillis(30),
            getNanosByMillis(40),
            getNanosByMillis(50),
            getNanosByMillis(70),
            getNanosByMillis(90),
            getNanosByMillis(110),
            getNanosByMillis(130),
            getNanosByMillis(150),
            getNanosByMillis(250),
            getNanosByMillis(300),
            getNanosByMillis(350),
            getNanosByMillis(400),
            getNanosByMillis(500),
            getNanosByMillis(600),
            getNanosByMillis(700),
            getNanosByMillis(1000),
            getNanosByMillis(2000),
            getNanosByMillis(5000),
            getNanosByMillis(10000)};

    private static long getNanosByMillis(long ms) {
        return Duration.ofMillis(ms).toNanos();
    }

    private MeterFilter meterFilter() {
        return new MeterFilter() {
            @Override
            public DistributionStatisticConfig configure(Meter.Id id, DistributionStatisticConfig config) {
                if (id.getType() == Meter.Type.TIMER && id.getName().contains("time")) {
                    return DistributionStatisticConfig.builder()
                            .sla(histBucketsBounds)
                            .build()
                            .merge(config);
                }
                return config;
            }
        };
    }

    private NamingConvention namingConvention() {
        return new NamingConvention() {
            @Override
            public String name(String name, Meter.Type type, String baseUnit) {
                return normalize(name);
            }

            @Override
            public String tagKey(String key) {
                return normalize(key);
            }

            private String normalize(String value) {
                return value.replace('-', '_').toLowerCase();
            }
        };
    }

    @Bean
    YasmMeterRegistry yasmMeterRegistry() {
        val registry = new YasmMeterRegistry();
        registry.config().namingConvention(namingConvention()).meterFilter(meterFilter());
        return registry;
    }

    @Primary
    @Bean
    CompositeMeterRegistry compositeMeterRegistry() {
        val compositeRegistry = new CompositeMeterRegistry();
        compositeRegistry.add(yasmMeterRegistry());
        return compositeRegistry;
    }
}
