package ru.yandex.reminders.boot;

import org.apache.log4j.AsyncAppender;
import org.apache.log4j.FileAppender;
import org.apache.log4j.Logger;
import org.springframework.context.ConfigurableApplicationContext;
import ru.yandex.misc.log.log4j.appender.ReopenOnHupFileAppender;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.main.MainSupport;
import ru.yandex.misc.property.PropertiesHolder;
import ru.yandex.misc.property.eval.EvalRecurrentProperties;
import ru.yandex.misc.property.load.strategy.PropertiesLoadStrategy;
import ru.yandex.misc.version.AbstractVersion;
import ru.yandex.reminders.api.ApiAppContextConfiguration;
import ru.yandex.reminders.log.TskvLogPatternLayout;
import ru.yandex.reminders.util.Configuration;
import ru.yandex.reminders.worker.WorkerAppContextConfiguration;

import java.util.TimeZone;

/**
 * @author dbrylev
 */
public class Main extends MainSupport {
    private static final ru.yandex.misc.log.mlf.Logger logger = LoggerFactory.getLogger(Main.class);

    public static final String ROOT_PACKAGE_PATH = "ru/yandex/reminders";

    private final RemindersAppName appName;
    private final boolean forTests;

    public Main(RemindersAppName appName, boolean forTests) {
        this.appName = appName;
        this.forTests = forTests;
    }

    public static void main(String args[]) {
        RemindersAppName appName = RemindersAppName.R.valueOfO(args[0]).getOrThrow("wrong application name");

        Configuration.initializeEnvironment();
        new Main(appName, false).runMain(args);
    }

    private static String evalProperty(String name) {
        return EvalRecurrentProperties.getProperty(PropertiesHolder.properties(), name);
    }

    @Override
    protected Class[] applicationContextPath() {
        if (appName == RemindersAppName.API) {
            return new Class[]{ApiAppContextConfiguration.class};
        } else if (appName == RemindersAppName.WORKER) {
            return new Class[]{WorkerAppContextConfiguration.class};
        } else {
            throw new IllegalArgumentException("wrong application name");
        }
    }

    @Override
    public void initializeLogger() {
        super.initializeLogger();

        TimeZone tz = TimeZone.getDefault();
        TimeZone.setDefault(TimeZone.getTimeZone("Europe/Moscow"));

        installTskvAppender();
        TimeZone.setDefault(tz);
    }

    @Override
    protected void initializeAppSpecific() throws Exception {
        super.initializeAppSpecific();
        logger.info("initializeAppSpecific for reminders");
    }

    private void installTskvAppender() {
        FileAppender fileAppender = new ReopenOnHupFileAppender();
        fileAppender.setFile(evalProperty("log.tskv.output"));
        fileAppender.setLayout(new TskvLogPatternLayout(evalProperty("log.tskv.format")));
        fileAppender.activateOptions();

        AsyncAppender asyncAppender = new AsyncAppender();
        asyncAppender.addAppender(fileAppender);

        Logger logger = Logger.getLogger("tskv");
        logger.setAdditivity(false);
        logger.addAppender(asyncAppender);
    }

    @Override
    protected AbstractVersion getVersion() {
        return RemindersAppVersion.VERSION;
    }

    @Override
    protected boolean isDaemon() {
        return true;
    }

    @Override
    protected boolean isCorbaSupported() {
        return false;
    }

    @Override
    protected void postProcessApplicationContextBeforeRefresh(ConfigurableApplicationContext context) {
        super.postProcessApplicationContextBeforeRefresh(context);

        context.addBeanFactoryPostProcessor(beanFactory -> beanFactory.registerSingleton("appName", appName));
    }

    @Override
    protected PropertiesLoadStrategy propertiesLoadStrategy() {
        return new RemindersPropertiesLoadStrategy(appName, forTests);
    }

}
