package ru.yandex.reminders.log;

import org.apache.log4j.spi.LoggingEvent;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.log4j.YandexPatternLayout;

import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class TskvLogPatternLayout extends YandexPatternLayout {
    private static final String PATTERN_TEMPLATE = "tskv"
            + "\ttskv_format=%s"
            + "\ttimestamp=%%d{yyyy-MM-dd HH:mm:ss}"
            + "\ttimezone=%%d{XX}"
            + "%%X{ndc}"
            + "\t%%m";

    private static final Pattern NDC_FIELD_PATTERN = Pattern.compile("#([^=]+)=([^#]+)(,#|$)");

    public TskvLogPatternLayout(String tskvFormat) {
        super(String.format(PATTERN_TEMPLATE, tskvFormat));
    }

    @Override
    public boolean ignoresThrowable() {
        return false;
    }

    @Override
    public String format(LoggingEvent event) {
        LoggingEvent modifiedEvent = new LoggingEvent(event.getFQNOfLoggerClass(),
                event.getLogger(), event.getTimeStamp(), event.getLevel(),
                escape(event.getRenderedMessage()),
                event.getThreadName(), event.getThrowableInformation(),
                decomposeNdc(event.getNDC()),
                event.getLocationInformation(),
                changeNdcCopyInMdcProperties(event.getProperties()));

        StringBuilder result = new StringBuilder(super.format(modifiedEvent));

        String[] stringRepresentation = event.getThrowableStrRep();
        if (stringRepresentation != null && stringRepresentation.length > 0) {
            result.append("\\n");
            for (String part : stringRepresentation) {
                result.append(escape(part)).append("\\n");
            }
        }
        result.append("\n");

        return result.toString();
    }

    @SuppressWarnings("unchecked")
    private Map changeNdcCopyInMdcProperties(Map mdcProperties) {
        if (mdcProperties.containsKey("ndc")) {
            Map changedProperties = new HashMap(mdcProperties);
            changedProperties.put("ndc", decomposeNdc((String) changedProperties.get("ndc")));
            return changedProperties;
        }
        return mdcProperties;
    }

    private String decomposeNdc(String ndc) {
        if (StringUtils.isNotBlank(ndc)) {
            Matcher matcher = NDC_FIELD_PATTERN.matcher(ndc);
            StringBuilder decomposedNdc = new StringBuilder();
            int currenPosition = 0;
            while (matcher.find(currenPosition)) {
                decomposedNdc
                        .append("\t")
                        .append(matcher.group(1))
                        .append("=")
                        .append(matcher.group(2));
                currenPosition = matcher.end() - 2;
            }
            return decomposedNdc.toString();
        }
        return ndc;
    }

    private String escape(String string) {
        string = string.replace("\\", "\\\\");

        return string
                .replace("\n", "\\n")
                .replace("\r", "\\r")
                .replace("\0", "\\0")
                .replace("\"", "\\\"");
    }
}
