package ru.yandex.reminders.logic.callback;

import java.io.Closeable;

import javax.annotation.PreDestroy;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.thread.WhatThreadDoes;
import ru.yandex.reminders.logic.event.EventId;
import ru.yandex.reminders.logic.reminder.Reminder;
import ru.yandex.reminders.logic.reminder.SendStatus;

@Slf4j
public class CallbackManager implements Closeable {
    private final HttpClient httpClient;

    public CallbackManager(int maxCons, Timeout timeout, ListF<HttpRequestInterceptor> interceptors) {
        val builder = ApacheHttpClientUtils.Builder.create()
                .multiThreaded()
                .withHttpsSupport(EnvironmentType.PRODUCTION.isActive()
                        ? ApacheHttpClientUtils.HttpsSupport.ENABLED
                        : ApacheHttpClientUtils.HttpsSupport.TRUST_ALL)
                .withTimeout(timeout)
                .withMaxConnectionsPerRoute(maxCons)
                .withMaxConnectionsTotal(maxCons)
                .withUserAgent("Yandex.Reminders");

        this.httpClient = interceptors.foldLeft(builder, ApacheHttpClientUtils.Builder::withInterceptorLast).build();
    }

    @PreDestroy
    public void close() {
        ApacheHttpClientUtils.stopQuietly(httpClient);
    }

    public SendStatus invoke(EventId id, Reminder reminder) {
        val uid = id.getUid();
        String url = reminder.getUrl().get();
        try {
            HttpGet request = new HttpGet(url);
            execute(request, "invoking callback for uid=" + uid);
            return new SendStatus.Sent(Option.of(url));
        } catch (RuntimeException e) {
            log.error("invoking callback for eventId={} failed, error={}", id, e);
            return new SendStatus.TryAgain(e.getMessage(), Option.of(url));
        }
    }

    private void execute(HttpUriRequest request, String info) {
        val h = WhatThreadDoes.push(info);
        try {
            val httpResponse = ApacheHttpClientUtils.execute(request, httpClient, (response) -> response);

            val statusCode = httpResponse.getStatusLine().getStatusCode();
            if (HttpStatus.is2xx(statusCode)) {
                log.info("{} completed successfully, status code={}", info, statusCode);
            } else {
                throw new RuntimeException("request wasn't completed successfully, status code=" + statusCode);
            }
        } finally {
            h.popSafely();
        }
    }
}
