package ru.yandex.reminders.logic.callmeback;

import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.IOException;
import java.net.URI;
import java.util.Collection;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import io.micrometer.core.instrument.MeterRegistry;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.StreamEx;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.a3.action.A3Exception;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.reminders.api.ApiBender;
import ru.yandex.reminders.api.reminder.ChannelsData;
import ru.yandex.reminders.api.reminder.ReminderData;
import ru.yandex.reminders.api.reminder.ReminderInfo;
import ru.yandex.reminders.api.reminder.RemindersInfo;
import ru.yandex.reminders.logic.callmeback.in.CallbackRequest;
import ru.yandex.reminders.logic.callmeback.out.CallmebackRequest;
import ru.yandex.reminders.logic.callmeback.out.Context;
import ru.yandex.reminders.logic.callmeback.out.Data;
import ru.yandex.reminders.logic.callmeback.out.HttpContext;
import ru.yandex.reminders.logic.callmeback.out.ListResponse;
import ru.yandex.reminders.logic.callmeback.out.NotifyScheme;
import ru.yandex.reminders.logic.event.Event;
import ru.yandex.reminders.logic.event.EventData;
import ru.yandex.reminders.logic.event.EventId;
import ru.yandex.reminders.logic.reminder.Reminder;

@Slf4j
public class CallmebackManager implements Closeable {
    public static final String METRIC_NAME = "reminders.callmeback.request";
    private static final String INTERNAL_ERROR_METRIC_NAME = String.format("%s.internal.error", METRIC_NAME);
    private static final String TIME_METRIC_NAME = String.format("%s.time", METRIC_NAME);
    private final HttpClient httpClient;
    private final String callmebackUrl;
    private final String remindersUri;
    private final MeterRegistry meterRegistry;

    public CallmebackManager(int maxCons, Timeout timeout,
                             Collection<HttpRequestInterceptor> interceptors,
                             String callmebackUrl, String remindersUrl,
                             MeterRegistry meterRegistry) {
        val builder = ApacheHttpClientUtils.Builder.create()
                .multiThreaded()
                .withHttpsSupport(EnvironmentType.PRODUCTION.isActive()
                        ? ApacheHttpClientUtils.HttpsSupport.ENABLED
                        : ApacheHttpClientUtils.HttpsSupport.TRUST_ALL)
                .withTimeout(timeout)
                .withMaxConnectionsPerRoute(maxCons)
                .withMaxConnectionsTotal(maxCons)
                .withUserAgent("Yandex.Reminders")
                ;

        interceptors.forEach(builder::withInterceptorLast);
        builder.withInterceptorLast(new HttpResponseRegistryInterceptor(METRIC_NAME, meterRegistry));

        this.httpClient = builder.build();
        this.callmebackUrl = callmebackUrl;
        this.remindersUri = getUri(remindersUrl, StreamEx.of("callmeback", "send")).toString();
        this.meterRegistry = meterRegistry;
    }

    @Override
    public void close() {
        ApacheHttpClientUtils.stopQuietly(httpClient);
    }

    private CallmebackResponse execute(HttpUriRequest request, boolean readContent) {
        val start = System.currentTimeMillis();
        try {
            return ApacheHttpClientUtils.execute(request, httpClient, (response) -> getResponse(response, readContent));
        } catch (Exception e) {
            meterRegistry.counter(INTERNAL_ERROR_METRIC_NAME).increment();
            val msg = "Failed to execute request to callmeback service";
            log.error(msg, e);
            throw new A3Exception("callmeback-fail", msg);
        } finally {
            val end = System.currentTimeMillis();
            meterRegistry.timer(TIME_METRIC_NAME).record(end-start, TimeUnit.MILLISECONDS);
        }
    }

    private static CallmebackResponse getResponse(HttpResponse response, boolean readContent) throws IOException {
        val statusCode =response.getStatusLine().getStatusCode();
        if (readContent) {
            try (val baos = new ByteArrayOutputStream()) {
                response.getEntity().writeTo(baos);
                return new CallmebackResponse(statusCode, Optional.of(baos.toByteArray()));
            }
        }
        return new CallmebackResponse(statusCode, Optional.empty());
    }

    private ListResponse getResponse(URI listUri) {
        val get = new HttpGet(listUri);
        val response = execute(get, true);
        val statusCode = response.getStatusCode();

        if (statusCode != HttpStatus.SC_OK) {
            val msg = String.format("Failed to find event(-s). Callmeback status code: %d", statusCode);
            throw new A3Exception("callmeback-fail", msg);
        }

        return ApiBender.mapper.parseJson(ListResponse.class, response.getContent().get());
    }

    private ListResponse getListResponseForSingleEvent(EventId id) {
        val groupKey = getGroupKey(id);

        val listUri = getBuilder(callmebackUrl, StreamEx.of("v1", "find", groupKey)).addParam("event_key_prefix", id.getExtId()).build();
        return getResponse(listUri);
    }

    private ListResponse getListResponseForClientEvents(PassportUid uid, String cid) {
        val groupKey = getGroupKey(uid, cid);
        val listUri = getCallmebackUri(StreamEx.of(groupKey));
        return getResponse(listUri);
    }

    public RemindersInfo listEvents(PassportUid uid, String cid) {
        val response = getListResponseForClientEvents(uid, cid);
        val map = StreamEx.of(response.getData())
                .map(Data::getContext)
                .map(Context::getContext)
                .map(HttpContext::getJson)
                .groupingBy(CallbackRequest::getExtId);

        val infos = StreamEx.of(map.values())
                .map(CallmebackManager::getReminderInfo)
                .flatMap(StreamEx::of)
                .toImmutableList();

        return new RemindersInfo(Cf.toList(infos), Option.empty(), Option.empty());
    }

    public Optional<ReminderInfo> findEvent(EventId id) {
        val response = getListResponseForSingleEvent(id);
        val callbackList = StreamEx.of(response.getData())
                .map(Data::getContext)
                .map(Context::getContext)
                .map(HttpContext::getJson)
                .toImmutableList();

        return getReminderInfo(callbackList)
                .filter(info -> info.getId().equals(id.getExtId()));
    }

    private static Optional<ReminderInfo> getReminderInfo(Collection<CallbackRequest> list) {
        if (list.isEmpty()) {
            return Optional.empty();
        }

        val first = list.stream().findFirst().get();
        val eventId = first.getId();
        val name = first.getData().getName();
        val desc = first.getData().getDescription();
        val sendDate = Option.of(first.getReminder().getSendDate());
        val channels = ChannelsData.create(StreamEx.of(list).map(CallbackRequest::getReminder).toImmutableList());
        val data = first.getData().getData();
        val reminderData = new ReminderData(name, desc, sendDate, channels, data);

        return Optional.of(new ReminderInfo(eventId.getExtId(), eventId.getCid(), reminderData));
    }

    public void removeEvent(EventId id) {
        log.debug("Removing event {}...", id);

        val response = getListResponseForSingleEvent(id);
        val groupKey = getGroupKey(id);

        response.getData().forEach(data -> removeEvent(groupKey, data));
    }

    private void removeEvent(String groupKey, Data data) {
        val eventKey = data.getEventKey();
        log.debug("Removing specific event {}/{}", groupKey, eventKey);
        val deleteUri = getCallmebackUri(StreamEx.of("delete", groupKey, eventKey));
        val post = new HttpPost(deleteUri);
        execute(post, false);
    }

    public void createOrUpdateEvent(EventId eventId, EventData data, Optional<String> senderName) {
        removeEvent(eventId);
        createEvent(eventId, data, senderName);
    }

    public void createEvent(EventId eventId, EventData data, Optional<String> senderName) {
        log.debug("Creating event {}...", eventId);
        val groupKey = getGroupKey(eventId);

        try {
            for (val reminder : data.getReminders()) {
                val request = getCallmebackRequest(eventId, reminder, data, senderName);
                val eventKey = eventId.getExtId() + "_" + reminder.getChannel().name();
                val response = createEvent(request, groupKey, eventKey);
                if (response.getStatusCode() != HttpStatus.SC_OK) {
                    throw new A3Exception("callmeback-fail", String.format("Failed to create event. Response code is: %d",
                            response.getStatusCode()));
                }
            }
        } catch (A3Exception e) {
            throw e;
        } catch (Exception e) {
            log.error(String.format("Error while creating event %s", eventId), e);
            removeEvent(eventId);
        }
    }

    private CallmebackRequest getCallmebackRequest(EventId eventId, Reminder reminder, EventData data, Optional<String> senderName) {
        val event = new Event(eventId, data, Option.x(senderName), Instant.now(), "callmeback");
        val callbackRequest = CallbackRequest.create(event, reminder);
        return new CallmebackRequest(reminder.getSendDate().withZone(DateTimeZone.UTC), remindersUri,
                NotifyScheme.HTTP, new Context(new HttpContext(HttpMethod.POST, callbackRequest)));
    }

    private CallmebackResponse createEvent(CallmebackRequest request, String groupKey, String eventKey) {
        log.debug("Creating reminder {}/{}", groupKey, eventKey);

        val uri = getCallmebackUri(StreamEx.of("add", groupKey, eventKey));
        val post = new HttpPost(uri);
        try {
            post.setEntity(new ByteArrayEntity(
                    ApiBender.mapper.serializeJson(request), ContentType.APPLICATION_JSON));
        } catch (Exception e) {
            val msg = "Problem while generating json request";
            log.error(msg, e);
            throw new A3Exception("reminder-fail", msg);
        }

        return execute(post, false);
    }

    private static String getGroupKey(EventId id) {
        return getGroupKey(id.getUid(), id.getCid());
    }

    private static String getGroupKey(PassportUid uid, String cid) {
        return String.format("%s-%s", uid, cid);
    }

    private static URI getUri(String url, StreamEx<String> appendPath) {
        return getBuilder(url, appendPath).build();
    }

    private static UriBuilder getBuilder(String url, StreamEx<String> appendPath) {
        val builder = UriBuilder.cons(url);
        appendPath.forEach(builder::appendPath);
        return builder;
    }

    private URI getCallmebackUri(StreamEx<String> appendPath) {
        return getUri(callmebackUrl, appendPath.prepend("v1", "event"));
    }
}