package ru.yandex.reminders.logic.cloudApi;

import lombok.val;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.util.EntityUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.tvm2.AddServiceTicketInterceptor;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.serialize.BenderJsonSerializer;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.reminders.logic.flight.FlightEventMeta;
import ru.yandex.reminders.tvm.TvmClient;
import ru.yandex.reminders.util.HostnameUtils;

import javax.annotation.PreDestroy;

public class CloudApiClient {
    private final BenderJsonSerializer<Flight> flightJsonizer = Bender.jsonSerializer(
            Flight.class, CloudApiBender.getConfiguration());
    private final HttpClient httpClient;
    private final String host;

    @Autowired
    private TvmClient tvmClient;
    @Value("${tvm.drive.alias:-drive}")
    private String driveAlias;

    public CloudApiClient(String host, Timeout timeout, int maxConnections) {
        HostnameUtils.validateHostOrHostPort(host);

        this.httpClient = ApacheHttpClientUtils.Builder.create()
                .multiThreaded()
                .withTimeout(timeout)
                .withHttpsSupport(ApacheHttpClientUtils.HttpsSupport.ENABLED)
                .withMaxConnections(maxConnections)
                .withInterceptorLast(new AddServiceTicketInterceptor(request -> getServiceTicketSupplier()))
                .build();
        this.host = host;
    }

    private Option<String> getServiceTicketSupplier() {
        return Option.of(tvmClient.getServiceTicketFor(driveAlias));
    }

    @PreDestroy
    public void close() {
        ApacheHttpClientUtils.stopQuietly(httpClient);
    }

    public void sendFlight(PassportUid uid, String id, FlightEventMeta flight) {
        val request = new HttpPut(flightUrl(uid, id));

        request.setEntity(new ByteArrayEntity(
                flightJsonizer.serializeJson(new Flight(flight)), ContentType.APPLICATION_JSON));

        execute(request, Cf.list());
    }

    public void deleteFlight(PassportUid uid, String id) {
        execute(new HttpDelete(flightUrl(uid, id)), Cf.list(404));
    }

    private String flightUrl(PassportUid uid, String id) {
        return String.format("https://%s/v1/%d/personality/profile/events/flights/actual/%s", host, uid.getUid(), id);
    }

    private void execute(HttpUriRequest request, ListF<Integer> non200AcceptableStatusCodes) {
        ApacheHttpClientUtils.execute(request, httpClient, response -> {
            val code = response.getStatusLine().getStatusCode();

            if (!HttpStatus.is2xx(code) && !non200AcceptableStatusCodes.containsTs(code)) {
                val body = EntityUtils.toString(response.getEntity());
                throw new HttpException(code, "Unexpected response status code: " + code + ", body: " + body);
            }
            return null;
        });
    }
}