package ru.yandex.reminders.logic.event;

import java.util.Optional;

import lombok.val;
import org.bson.types.ObjectId;
import org.joda.time.Instant;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.mongo3.bender.MongoId;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderFlatten;
import ru.yandex.misc.bender.annotation.BenderMembersToBind;
import ru.yandex.reminders.api.reminder.Source;
import ru.yandex.reminders.logic.flight.FlightEventMeta;
import ru.yandex.reminders.logic.reminder.Channel;
import ru.yandex.reminders.logic.reminder.Reminder;
import ru.yandex.reminders.logic.reminder.ReminderWithEvent;

@Bendable
@BenderMembersToBind(value = MembersToBind.ALL_FIELDS)
public class Event {
    // this _id is only because of problems with indexing of compound EventId id
    // see http://stackoverflow.com/questions/7246434/expected-behaviour-of-compound-id-in-mongodb
    // details: we don't want to waste space twice: for default _id_ index, and our per-component index
    // rest of code doesn't use this _id field
    @MongoId
    private final ObjectId _id;

    @BenderFlatten
    private final EventId id;
    @BenderFlatten
    private final EventData data;

    private final Option<String> sender;

    private final Instant updTs;
    private final String updReq;

    public Event(EventId id, EventData data, Option<String> senderName, Instant updTs, String updReq) {
        this._id = ObjectId.get();
        this.id = id;
        this.data = data;
        this.sender = senderName;
        this.updTs = updTs;
        this.updReq = updReq;
    }

    public static Function<Event, String> getClientIdF() {
        return e -> e.getClientId();
    }

    public static Function<Event, Option<FlightEventMeta>> getFlightMetaF() {
        return e -> e.getFlightMeta();
    }

    public static Function<Event, ListF<Reminder>> getRemindersF() {
        return e -> e.getReminders();
    }

    public static Function<Event, ListF<ReminderWithEvent>> getRemindersWithEventF() {
        return e -> e.getRemindersWithEvent();
    }

    public ObjectId getMongoId() {
        return _id;
    }

    public EventId getId() {
        return id;
    }

    public PassportUid getUid() {
        return id.getUid();
    }

    public String getClientId() {
        return id.getCid();
    }

    public String getExternalId() {
        return id.getExtId();
    }

    public long getIdx() {
        return id.getIdx();
    }

    public Option<String> getName() {
        return data.getName();
    }

    public Option<String> getDescription() {
        return data.getDescription();
    }

    public Option<JsonObject> getData() {
        return data.getData();
    }

    public EventData getEventData() {
        return data;
    }

    public Option<FlightEventMeta> getFlightMeta() {
        return data.getFlightMeta();
    }

    public ListF<Reminder> getReminders() {
        return data.getReminders();
    }

    public ListF<Channel> getReminderChannels() {
        return getReminders().map(Reminder.getChannelF()).stableUnique();
    }

    public ListF<ReminderWithEvent> getRemindersWithEvent() {
        return getReminders().map(ReminderWithEvent.consF().bind2(this));
    }

    public Option<Source> getSource() {
        return data.getSource();
    }

    public Option<String> getSenderName() {
        return sender;
    }

    public Instant getUpdatedTs() {
        return updTs;
    }

    public String getUpdatedReqId() {
        return updReq;
    }

    public boolean isFlight() {
        return SpecialClientIds.FLIGHT.equals(id.getCid());
    }

    public boolean isTv() {
        return SpecialClientIds.TV.equals(id.getCid());
    }

    public boolean isHotel() {
        return SpecialClientIds.HOTEL.equals(id.getCid());
    }

    public static Optional<String> extractMid(Optional<FlightEventMeta> meta, EventData data) {
        val result = meta.map(FlightEventMeta::getMid);
        if (result.isPresent()) {
            return result;
        }
        return EventDataUtils.extractMid(data.getData().cast()).toOptional();
    }
}