package ru.yandex.reminders.logic.event;

import com.mongodb.client.model.Filters;
import org.bson.types.ObjectId;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import ru.yandex.bolts.collection.*;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.bolts.function.Function2;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.scheduler.TaskCategory;
import ru.yandex.commune.mongo3.FiltersX;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.reminders.logic.reminder.Channel;
import ru.yandex.reminders.logic.reminder.Reminder;
import ru.yandex.reminders.logic.reminder.ReminderWithEvent;
import ru.yandex.reminders.logic.update.ActionInfo;
import ru.yandex.reminders.worker.ReminderSendTask;
import ru.yandex.reminders.worker.ReminderSendTaskParameters;

public class EventManager {
    private static final Logger logger = LoggerFactory.getLogger(EventManager.class);

    @Autowired
    private EventMdao eventMdao;
    @Autowired
    private BazingaTaskManager bazingaTaskManager;

    private static Event internalCreateEvent(
            EventId id, EventData data, Option<String> senderName, ActionInfo actionInfo) {
        Function<Reminder, ObjectId> idF = newObjectIdF().asFunction();
        ListF<Reminder> reminders = data.getReminders().map(Reminder.withIdF(idF));

        data = data.withReminders(reminders);

        return new Event(id, data, senderName, actionInfo.getNow(), actionInfo.getRequestIdWithHostId());
    }

    private static ObjectId newObjectId() {
        return ObjectId.get();
    }

    private static Function0<ObjectId> newObjectIdF() {
        return EventManager::newObjectId;
    }

    public Option<Event> findEvent(EventId id) {
        return eventMdao.findEvent(id);
    }

    public ListF<Event> findEvents(PassportUid uid, String clientId, EventsFilter filter) {
        return eventMdao.findEvents(filter.toMongoQuery(uid, clientId), filter.getOrder(), filter.getLimits());
    }

    public long countEvents(PassportUid uid, String clientId, EventsFilter filter) {
        return eventMdao.countEvents(filter.toMongoQuery(uid, clientId));
    }

    public ListF<Event> findEventsByReminderSendTs(PassportUid uid, ListF<String> clientIds, InstantInterval interval) {
        return eventMdao.findEvents(Filters.and(
                Filters.eq("uid", uid.getUid()),
                Filters.in("cid", clientIds),
                FiltersX.between("reminders.sendTs", interval)));
    }

    public void deleteEvents(PassportUid uid, String clientId, EventsFilter filter) {
        eventMdao.deleteEvents(filter.toMongoQuery(uid, clientId));
    }

    public void deleteReminders(PassportUid uid, String clientId, EventsFilter filter, Channel channel) {
        eventMdao.deleteReminders(filter.toMongoQuery(uid, clientId), Filters.eq("channel", channel.value()));
    }

    public void deleteReminders(PassportUid uid, String clientId, EventsFilter filter, ListF<ObjectId> reminderIds) {
        eventMdao.deleteReminders(filter.toMongoQuery(uid, clientId), Filters.in("id", reminderIds));
    }

    public Event createOrUpdateEvent(EventId id, EventData data, ActionInfo actionInfo) {
        return createOrUpdateEvent(id, data, Option.empty(), actionInfo);
    }

    public Event createOrUpdateEvent(EventId id, EventData data, Option<String> senderName, ActionInfo actionInfo) {
        Event event = internalCreateEvent(id, data, senderName, actionInfo);
        scheduleRemindersSending(event, actionInfo);
        eventMdao.saveOrUpdateEvent(event);
        return event;
    }

    public Event updateEvent(Event event, EventData data, ActionInfo actionInfo) {
        return updateEvents(Tuple2List.fromPairs(event, data), actionInfo).single();
    }

    public ListF<Event> updateEvents(Tuple2List<Event, EventData> updates, ActionInfo actionInfo) {
        ListF<Event> events = Cf.arrayList();

        for (Tuple2<Event, EventData> update : updates) {
            EventData data = update.get2();

            Function<Reminder, ObjectId> idF = newObjectIdF().asFunction();
            ListF<Reminder> reminders = data.getReminders().map(Reminder.withIdF(idF));

            data = data.withReminders(reminders);

            events.add(new Event(
                    update.get1().getId(), data, update.get1().getSenderName(),
                    actionInfo.getNow(), actionInfo.getRequestIdWithHostId()));
        }
        scheduleRemindersSending(events, actionInfo);

        for (Event event : events) {
            eventMdao.updateEvent(event);
        }
        return events;
    }

    public Function2<EventId, EventData, Event> createOrUpdateEventF(final ActionInfo actionInfo) {
        return (eventId, eventData) -> createOrUpdateEvent(eventId, eventData, Option.<String>none(), actionInfo);
    }

    private void scheduleRemindersSending(ListF<Event> events, ActionInfo actionInfo) {
        events.flatMap(Event.getRemindersWithEventF()).forEach(scheduleReminderSendingF(actionInfo));
    }

    private void scheduleRemindersSending(Event event, ActionInfo actionInfo) {
        event.getRemindersWithEvent().forEach(scheduleReminderSendingF(actionInfo));
    }

    private void scheduleReminderSending(ReminderWithEvent rem, final ActionInfo actionInfo) {
        Instant sendTs = rem.getReminder().getSendTs();

        if (sendTs.isBefore(actionInfo.getNow())) {
            long minutesBefore = new Duration(sendTs, actionInfo.getNow()).getStandardMinutes();

            logger.debug("Skip event {} reminder {} scheduling in {} minutes before now",
                    rem.getEvent().getId(), rem.getReminder().getId(), minutesBefore);
            return;
        }

        ReminderSendTaskParameters parameters = new ReminderSendTaskParameters(
                rem.getEvent().getId(), rem.getReminder().getId(),
                actionInfo.getNow(), actionInfo.getRequestIdWithHostId());

        schedule(parameters, sendTs);
    }

    private void schedule(ReminderSendTaskParameters parameters, Instant sendTs) {
        ReminderSendTask task = new ReminderSendTask(parameters);
        bazingaTaskManager.schedule(task, TaskCategory.DEFAULT, sendTs, task.priority(), true);
    }

    private Function1V<ReminderWithEvent> scheduleReminderSendingF(final ActionInfo actionInfo) {
        return r -> scheduleReminderSending(r, actionInfo);
    }
}
