package ru.yandex.reminders.logic.event;

import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.Updates;
import io.micrometer.core.instrument.MeterRegistry;
import org.bson.conversions.Bson;
import org.bson.types.ObjectId;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.commune.mongo3.schema.IndexInfo;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.db.q.SqlOrder;
import ru.yandex.reminders.logic.flight.FlightSource;
import ru.yandex.reminders.logic.reminder.Reminder;
import ru.yandex.reminders.mongodb.AbstractMdao;

public class EventMdao extends AbstractMdao<ObjectId, Event> {
    private final static String SAVED_EVENTS_METRIC = "reminders.saved.events";
    private final static String SAVED_REMINDERS_METRIC = "reminders.saved.reminders";
    private final static String SAVED_FLIGHT_TICKETS_METRIC = "reminders.saved.flight.tickets";

    @Autowired
    private MeterRegistry registry;

    public EventMdao(MongoDatabase remindersDb) {
        super(remindersDb, "event", Event.class);
    }

    @Override
    public ListF<IndexInfo> getIndexes() {
        return Cf.list(
                new IndexInfo().key("uid", 1).key("cid", 1).key("extId", 1).key("idx", 1).unique(true),
                new IndexInfo().key("uid", 1).key("reminders.sendTs", 1),
                new IndexInfo()
                        .key("flight.num", 1).key("flight.depGeoId", 1)
                        .key("flight.planDepTs", 1).key("flight.source", 1));
    }

    public Option<Event> findEvent(EventId id) {
        return collectionX.findOne(toQuery(id));
    }

    public ListF<Event> findEvents(Bson query) {
        return collectionX.find(query);
    }

    public ListF<Event> findEvents(Bson query, SqlOrder order, SqlLimits limits) {
        return collectionX.find(query, order, limits);
    }

    public long countEvents(Bson query) {
        return collectionX.count(query);
    }

    public ListF<Event> findFlightEvents(CollectionF<PassportUid> excludeUids, String flightNumber, int depGeoId, Instant plannedTs, FlightSource source) {
        ListF<Bson> filters = Cf.arrayList();

        if (excludeUids.isNotEmpty()) {
            filters.add(Filters.nin("uid", excludeUids.map(AbstractMdao::toMongoValue)));
        }
        filters.add(Filters.eq("flight.num", flightNumber));
        filters.add(Filters.eq("flight.depGeoId", depGeoId));
        filters.add(Filters.eq("flight.planDepTs", toMongoValue(plannedTs)));
        filters.add(Filters.eq("flight.source", toMongoValue(source)));

        return collectionX.find(Filters.and(filters));
    }

    public void saveOrUpdateEvent(Event e) {
        collectionX.replaceOneWithoutId(toQuery(e.getId()), e);

        final String clientId = SpecialClientIds.isSpecial(e.getClientId()) ? e.getClientId() : "OTHERS";
        registry.counter(SAVED_EVENTS_METRIC).increment();
        e.getReminders().forEach((Function1V<Reminder>) r -> {
            String channel = r.getChannel().toString().toLowerCase();
            registry.counter(SAVED_REMINDERS_METRIC + "." + clientId + "." + channel).increment();
        });
        if (e.getFlightMeta().isDefined()) {
            Option<FlightSource> sourceO = e.getFlightMeta().get().getSource();
            String src = sourceO.isDefined() ? sourceO.get().name().toLowerCase() : "none";
            registry.counter(SAVED_FLIGHT_TICKETS_METRIC + "." + src).increment();
        }
    }

    public void updateEvent(Event event) {
        collectionX.replaceOneWithoutId(toQuery(event.getId()), event);
    }

    public void deleteEvents(Bson query) {
        collectionX.deleteMany(query);
    }

    public void deleteReminders(Bson eventQuery, Bson reminderQuery) {
        collectionX.updateMany(eventQuery, Updates.pullByFilter(Filters.eq("reminders", reminderQuery)));
    }

    public static Bson toQuery(EventId id) {
        return Filters.and(
                Filters.eq("uid", id.getUid().getUid()),
                Filters.eq("cid", id.getCid()),
                Filters.eq("extId", id.getExtId()),
                Filters.eq("idx", id.getIdx()));
    }
}
