package ru.yandex.reminders.logic.event;

import com.mongodb.client.model.Filters;
import org.bson.conversions.Bson;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.mongo3.FiltersX;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.db.q.SqlOrder;
import ru.yandex.misc.time.InstantInterval;

public class EventsFilter {
    private final Option<ListF<String>> externalIdFilter;
    private final Option<ListF<Long>> idxFilter;
    private final Option<InstantInterval> reminderSendTsFilter;
    private final Option<ListF<Long>> notInIdxsFilter;

    private final Option<IterationKey> prevId;
    private final Order order;
    private final SqlLimits limits;

    private EventsFilter(
            Option<ListF<String>> externalIdFilter,
            Option<ListF<Long>> idxFilter,
            Option<InstantInterval> reminderSendTsFilter,
            Option<ListF<Long>> notInIdxsFilter,
            Option<IterationKey> prevId,
            Order order, SqlLimits limits) {
        this.externalIdFilter = externalIdFilter;
        this.idxFilter = idxFilter;
        this.reminderSendTsFilter = reminderSendTsFilter;
        this.notInIdxsFilter = notInIdxsFilter;
        this.prevId = prevId;
        this.order = order;
        this.limits = limits;
    }

    public static EventsFilter any() {
        return new EventsFilter(
                Option.empty(), Option.empty(), Option.empty(),
                Option.empty(), Option.empty(), Order.UNORDERED, SqlLimits.all());
    }

    public static EventsFilter byExternalId(String externalId) {
        return any().andByExternalId(externalId);
    }

    public EventsFilter andByExternalId(ListF<String> externalIds) {
        return new EventsFilter(
                Option.of(externalIds), idxFilter, reminderSendTsFilter, notInIdxsFilter, prevId, order, limits);
    }

    public EventsFilter andByIdx(ListF<Long> idxs) {
        return new EventsFilter(
                externalIdFilter, Option.of(idxs), reminderSendTsFilter, notInIdxsFilter, prevId, order, limits);
    }

    public EventsFilter andByReminderSendTs(InstantInterval interval) {
        return new EventsFilter(
                externalIdFilter, idxFilter, Option.of(interval), notInIdxsFilter, prevId, order, limits);
    }

    public EventsFilter andByIdxNotIn(ListF<Long> idxs) {
        return new EventsFilter(
                externalIdFilter, idxFilter, reminderSendTsFilter, Option.of(idxs), prevId, order, limits);
    }

    public EventsFilter andWithPrevId(IterationKey id) {
        return new EventsFilter(
                externalIdFilter, idxFilter, reminderSendTsFilter, notInIdxsFilter, Option.of(id), order, limits);
    }

    public EventsFilter withOrder(Order order) {
        return new EventsFilter(
                externalIdFilter, idxFilter, reminderSendTsFilter, notInIdxsFilter, prevId, order, limits);
    }

    public EventsFilter withLimits(SqlLimits limits) {
        return new EventsFilter(
                externalIdFilter, idxFilter, reminderSendTsFilter, notInIdxsFilter, prevId, order, limits);
    }

    public EventsFilter andByExternalId(String externalId) {
        return andByExternalId(Cf.list(externalId));
    }

    public Bson toMongoQuery(PassportUid uid, String clientId) {
        ListF<Bson> filters = Cf.arrayList();

        filters.add(Filters.eq("uid", uid.getUid()));
        filters.add(Filters.eq("cid", clientId));

        externalIdFilter.forEach(ids -> filters.add(Filters.in("extId", ids)));
        reminderSendTsFilter.forEach(interval -> filters.add(FiltersX.between("reminders.sendTs", interval)));

        idxFilter.forEach(vs -> filters.add(Filters.in("idx", vs)));
        notInIdxsFilter.forEach(vs -> filters.add(Filters.nin("idx", vs)));

        prevId.forEach(id -> filters.add(order.toFilter(id)));

        return Filters.and(filters);
    }

    public SqlOrder getOrder() {
        return order.toSqlOrderFor(limits);
    }

    public SqlLimits getLimits() {
        return limits;
    }
}
