package ru.yandex.reminders.logic.flight;

import lombok.val;
import org.joda.time.*;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderMembersToBind;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.time.TimeUtils;
import ru.yandex.reminders.i18n.Language;
import ru.yandex.reminders.util.task.TimeZoneMigrationTask;

@Bendable
@BenderMembersToBind(value = MembersToBind.ALL_FIELDS)
public class FlightEventMeta extends DefaultObject {
    private final String mid;  // mail message id
    private final String num;  // flight number

    private final Option<FlightItem> airline;

    private final FlightCity depCity;
    private final Option<FlightItem> depAirport;
    private final Instant depTs;                  // actual departure datetime
    private final DateTimeZone depTz;             // actual departure timezone

    private final FlightCity arrCity;
    private final Option<FlightItem> arrAirport;
    private final Option<Instant> arrTs;
    private final Option<DateTimeZone> arrTz;

    private final Option<Instant> planDepTs;      // planned departure datetime
    private final Option<Integer> depGeoId;
    private final Option<FlightSource> source;          // Option only because we don't know source for old data

    private final Option<String> checkIn; // checkInLink
    private final Option<String> aeroexp; // aeroexpressLink

    private final Option<String> lastNum;          // lastSegmentFlightNumber
    private final Option<LocalDateTime> lastDepTs; // lastSegmentDepartureDateTime
    private final Option<FlightSource> lastSource;       // Option only because we don't know source for old data

    private final Option<Direction> dir;
    private final Option<String> dirStr;  // original direction as string, if enum parsing failed

    private final Option<String> lang;
    private final Option<String> yaDomain;

    private Option<Boolean> migrate = Option.of(TimeZoneMigrationTask.MIGRATION_FLAG);

    public FlightEventMeta(
            String mid, String flightNumber, Option<FlightItem> airline,
            Option<LocalDateTime> plannedDepartureDateTime,
            FlightCity departureCity, Option<FlightItem> departureAirport,
            LocalDateTime departureDateTime, DateTimeZone departureTz,
            FlightCity arrivalCity, Option<FlightItem> arrivalAirport,
            Option<LocalDateTime> arrivalDateTime, Option<DateTimeZone> arrivalTz,
            Option<FlightSource> source, Option<String> checkInLink, Option<String> aeroexpressLink,
            Option<String> lastSegmentFlightNumber, Option<LocalDateTime> lastSegmentDepartureDateTime,
            Option<FlightSource> lastSegmentSource, Option<String> direction,
            Option<String> lang, Option<String> yaDomain) {
        this.mid = mid;
        this.num = flightNumber;
        this.airline = airline;
        this.depCity = departureCity;
        this.depAirport = departureAirport;
        this.depTs = departureDateTime.toDateTime(departureTz).toInstant();
        this.depTz = departureTz;
        this.arrCity = arrivalCity;
        this.arrAirport = arrivalAirport;
        this.arrTs = arrivalDateTime.map(dt -> dt.toDateTime(arrivalTz.get()).toInstant());
        this.arrTz = arrivalTz;
        this.planDepTs = plannedDepartureDateTime.map(TimeUtils.localDateTime.toInstantF(departureTz));

        this.depGeoId = departureCity.getGeoId(); // XXX reindex db collection
        this.source = source;
        this.checkIn = checkInLink;
        this.aeroexp = aeroexpressLink;
        this.lastNum = lastSegmentFlightNumber;
        this.lastDepTs = lastSegmentDepartureDateTime;
        this.lastSource = lastSegmentSource;
        this.dir = direction.isDefined() ? Direction.R.valueOfO(direction.get()) : Option.<Direction>none();
        this.dirStr = this.dir.isEmpty() ? direction : Option.<String>none();
        this.lang = lang;
        this.yaDomain = yaDomain;
    }

    private FlightEventMeta(
            String mid, String num, Option<FlightItem> airline,
            FlightCity depCity, Option<FlightItem> depAirport, Instant depTs, DateTimeZone depTz,
            FlightCity arrCity, Option<FlightItem> arrAirport, Option<Instant> arrTs, Option<DateTimeZone> arrTz,
            Option<Instant> planDepTs, Option<Integer> depGeoId,
            Option<FlightSource> source, Option<String> checkIn, Option<String> aeroexp,
            Option<String> lastNum, Option<LocalDateTime> lastDepTs,
            Option<FlightSource> lastSource, Option<Direction> dir, Option<String> dirStr,
            Option<String> lang, Option<String> yaDomain) {
        this.mid = mid;
        this.num = num;
        this.airline = airline;
        this.depCity = depCity;
        this.depAirport = depAirport;
        this.depTs = depTs;
        this.depTz = depTz;
        this.arrCity = arrCity;
        this.arrAirport = arrAirport;
        this.arrTs = arrTs;
        this.arrTz = arrTz;
        this.planDepTs = planDepTs;
        this.depGeoId = depGeoId;
        this.source = source;
        this.checkIn = checkIn;
        this.aeroexp = aeroexp;
        this.lastNum = lastNum;
        this.lastDepTs = lastDepTs;
        this.lastSource = lastSource;
        this.dir = dir;
        this.dirStr = dirStr;
        this.lang = lang;
        this.yaDomain = yaDomain;
    }

    public static Function<FlightEventMeta, Instant> getDepartureTsF() {
        return FlightEventMeta::getDepartureTs;
    }

    public static Function<FlightEventMeta, String> getFlightNumberF() {
        return FlightEventMeta::getFlightNumber;
    }

    public String getMid() {
        return mid;
    }

    public String getFlightNumber() {
        return num;
    }

    public Option<FlightItem> getAirline() {
        return airline;
    }

    public FlightCity getDepartureCity() {
        return depCity;
    }

    public Option<FlightItem> getDepartureAirport() {
        return depAirport;
    }

    public DateTime getDepartureDateTime() {
        return depTs.toDateTime(depTz);
    }

    public DateTimeZone getDepartureCityTz() {
        return depTz;
    }

    public Instant getDepartureTs() {
        return depTs;
    }

    public FlightCity getArrivalCity() {
        return arrCity;
    }

    public Option<FlightItem> getArrivalAirport() {
        return arrAirport;
    }

    public Option<DateTime> getArrivalDateTime() {
        return arrTs.map(dt -> dt.toDateTime(arrTz.get()));
    }

    public Option<DateTimeZone> getArrivalTz() {
        return arrTz;
    }

    public Option<Instant> getArrivalTs() {
        return arrTs;
    }

    public Option<DateTime> getPlannedDepartureDateTime() {
        return planDepTs.map(TimeUtils.instant.dateTime(depTz));
    }

    public Option<Instant> getPlannedDepartureTs() {
        return planDepTs;
    }

    public LocalDateTime getDepartureLocalDateTime() {
        return getDepartureDateTime().toLocalDateTime();
    }

    public Option<String> getCheckInLink() {
        return checkIn;
    }

    public Option<String> getAeroexpressLink() {
        return aeroexp;
    }

    public Option<String> getLastSegmentFlightNumber() {
        return lastNum;
    }

    public Option<LocalDateTime> getLastSegmentDepartureDateTime() {
        return lastDepTs;
    }

    public Option<FlightSource> getLastSegmentSource() {
        return lastSource;
    }

    public Option<String> getDirection() {
        return dir.isPresent() ? Option.of(Direction.R.toXmlName(dir.get())) : dirStr;
    }

    public Option<FlightSource> getSource() {
        return source;
    }

    public Option<Integer> getDepGeoId() {
        return depGeoId;
    }

    public boolean isSegmented() {
        return lastNum.isDefined() || lastDepTs.isDefined() || lastSource.isDefined();
    }

    public Option<String> getLang() {
        return lang;
    }

    public Language getLanguage() {
        return Language.fromCodeOrRussian(lang);
    }

    public Option<String> getYaDomain() {
        return yaDomain;
    }

    public String getYaDomainOrYandexRu() {
        return yaDomain.getOrElse("yandex.ru");
    }

    public boolean isMigrated() {
        return migrate.isSome(TimeZoneMigrationTask.MIGRATION_FLAG);
    }

    public void setNotMigrated() {
        migrate = Option.none();
    }

    public FlightEventMeta withDepartureTs(Instant newDepTs) {
        val newArrTs = !isSegmented()
                ? arrTs.map(ts -> ts.plus(new Duration(depTs, newDepTs)))
                : arrTs;

        return new FlightEventMeta(
                mid, num, airline, depCity, depAirport, newDepTs, depTz, arrCity, arrAirport, newArrTs, arrTz,
                planDepTs, depGeoId, source, checkIn, aeroexp, lastNum, lastDepTs, source, dir, dirStr, lang, yaDomain);
    }
}
