package ru.yandex.reminders.logic.flight;

import java.util.Optional;

import lombok.val;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.scheduler.TaskCategory;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.sms.SmsPassportErrorCode;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.TimeUtils;
import ru.yandex.reminders.api.flight.FlightDataConverter;
import ru.yandex.reminders.api.flight.MailReminderData;
import ru.yandex.reminders.i18n.Language;
import ru.yandex.reminders.logic.callmeback.in.CallbackRequest;
import ru.yandex.reminders.logic.event.Event;
import ru.yandex.reminders.logic.event.EventData;
import ru.yandex.reminders.logic.event.EventId;
import ru.yandex.reminders.logic.event.EventManager;
import ru.yandex.reminders.logic.event.EventsFilter;
import ru.yandex.reminders.logic.event.SpecialClientIds;
import ru.yandex.reminders.logic.flight.shift.FlightShift;
import ru.yandex.reminders.logic.flight.shift.FlightShiftMdao;
import ru.yandex.reminders.logic.flight.shift.SendingTimeUtils;
import ru.yandex.reminders.logic.reminder.Channel;
import ru.yandex.reminders.logic.reminder.EventType;
import ru.yandex.reminders.logic.reminder.Origin;
import ru.yandex.reminders.logic.reminder.PhoneNumber;
import ru.yandex.reminders.logic.reminder.Reminder;
import ru.yandex.reminders.logic.reminder.ReminderSendManager;
import ru.yandex.reminders.logic.reminder.SendDailyStat;
import ru.yandex.reminders.logic.sending.SmsSender;
import ru.yandex.reminders.logic.update.ActionInfo;
import ru.yandex.reminders.logic.update.LockManager;
import ru.yandex.reminders.logic.user.SettingsManager;
import ru.yandex.reminders.util.DateTimeUtils;
import ru.yandex.reminders.worker.DeleteCloudApiFlightsTask;
import ru.yandex.reminders.worker.DeleteCloudApiFlightsTaskParameters;
import ru.yandex.reminders.worker.FlightShiftSendTask;
import ru.yandex.reminders.worker.FlightShiftSendTaskParameters;

public class FlightReminderManager {
    private static final Logger logger = LoggerFactory.getLogger(FlightReminderManager.class);

    @Autowired
    private LockManager lockManager;
    @Autowired
    private SmsSender smsSender;
    @Autowired
    private ReminderSendManager reminderSendManager;
    @Autowired
    private SettingsManager settingsManager;
    @Autowired
    private EventManager eventManager;
    @Autowired
    private FlightShiftMdao flightShiftMdao;
    @Autowired
    private BazingaTaskManager bazingaTaskManager;

    public FlightReminderManager() {
    }

    FlightReminderManager(BazingaTaskManager bazingaTaskManager) {
        this.bazingaTaskManager = bazingaTaskManager;
    }

    private static Function<FlightEventMeta, Tuple2<EventId, EventData>> toEventIdAndDataF(
            PassportUid uid, String externalId, MailReminderData mailReminderData, ActionInfo actionInfo) {
        return meta -> Tuple2.tuple(
                new EventId(
                        uid, SpecialClientIds.FLIGHT, externalId,
                        meta.getPlannedDepartureDateTime().getOrElse(meta.getDepartureDateTime())
                                .toLocalDateTime().toDateTime(DateTimeZone.UTC).getMillis()),
                new EventData(
                        Option.of(ru.yandex.reminders.api.reminder.Source.INTERNAL),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        FlightDataConverter.toReminders(meta, mailReminderData, actionInfo.getNow()),
                        Option.of(meta)
                )
        );
    }

    private static Option<MailReminderData> toMailReminderDataO(ListF<Event> events) {
        return events.firstO().map(Event.getRemindersF().andThen(toMailReminderDataF()));
    }

    // almost equivalent to old FlightReminders
    private static MailReminderData toMailReminderData(ListF<Reminder> reminders) {
        val smss = reminders.filter(Reminder.channelIsF(Channel.SMS));
        val emails = reminders.filter(Reminder.channelIsF(Channel.EMAIL));

        val userCreatedSmss = smss.filter(Reminder.originIsF(Origin.USER));
        val autoCreatedSmss = smss.filter(Reminder.originIsF(Origin.AUTO));

        return new MailReminderData(
                smss.filterMap(Reminder.getPhoneF()).unique().singleO(),
                singleOffset(autoCreatedSmss), singleOffset(userCreatedSmss),
                emails.filterMap(Reminder.getEmailF()).unique().singleO(),
                singleOffset(emails)
        );
    }

    private static Function<ListF<Reminder>, MailReminderData> toMailReminderDataF() {
        return FlightReminderManager::toMailReminderData;
    }

    private static Option<Duration> singleOffset(ListF<Reminder> reminders) {
        return reminders.filterMap(Reminder.getOffsetF()).unique().singleO().map(DateTimeUtils.minutesDurationF());
    }

    public FlightRemindersInfo getFlightRemindersInfo(
            PassportUid uid, String externalId, ListF<FlightEventMeta> flightEventMetas, ActionInfo actionInfo) {
        ListF<Event> events = findEvents(uid, externalId);
        return events.isEmpty() ? toFlightRemindersInfo(flightEventMetas, actionInfo) : toFlightRemindersInfo(
                toMailReminderDataO(events).getOrElse(MailReminderData.empty()),
                events.filterMap(Event.getFlightMetaF()),
                actionInfo
        );
    }

    public FlightRemindersInfo createOrUpdateFlightReminders(
            final PassportUid uid, final String externalId, final MailReminderData mailReminderData,
            final ListF<FlightEventMeta> flightEventMetas, final ActionInfo actionInfo) {
        return lockManager.withLock(uid, EventType.FLIGHT, new Function0<FlightRemindersInfo>() {
            public FlightRemindersInfo apply() {
                return doCreateOrUpdateFlightReminders(uid, externalId, mailReminderData, flightEventMetas, actionInfo);
            }
        });
    }

    public void deleteFlightReminders(PassportUid uid, String externalId, ActionInfo actionInfo) {
        lockManager.withLock(uid, EventType.FLIGHT, () -> {
            ListF<Event> events = findEvents(uid, externalId);
            ListF<Reminder> reminders = events.flatMap(Event::getReminders);

            DeleteCloudApiFlightsTask deletionTask = new DeleteCloudApiFlightsTask(
                    new DeleteCloudApiFlightsTaskParameters(uid, externalId, events.map(Event::getIdx)));

            bazingaTaskManager.schedule(deletionTask, actionInfo.getNow().plus(Duration.standardMinutes(10)));

            eventManager.deleteReminders(
                    uid, SpecialClientIds.FLIGHT,
                    EventsFilter.byExternalId(externalId), reminders.map(Reminder::getId));
        });
    }

    public void deleteSmsFlightReminders(final PassportUid uid, final String externalId) {
        lockManager.withLock(uid, EventType.FLIGHT, () -> eventManager.deleteReminders(
                uid, SpecialClientIds.FLIGHT, EventsFilter.byExternalId(externalId), Channel.SMS));
    }

    public void sendPromiseSms(PassportUid uid, Option<PhoneNumber> phone, Duration offset, Option<String> lang) {
        smsSender.sendToUser(uid, phone,
                FlightReminderSmsMessageCreator.createPromiseSmsMessage(offset, Language.fromCodeOrRussian(lang)),
                SpecialClientIds.FLIGHT);
    }

    public void disableFlightReminders(PassportUid uid, boolean disabled) {
        settingsManager.saveFlightRemindersDisabled(uid, disabled);
    }

    public ListF<SendDailyStat> getSmsFlightRemindersSendStats(boolean allHistory) { // CAL-6356
        LocalDate till = LocalDate.now(DateTimeZone.UTC).minusDays(1);
        Option<LocalDate> since = Option.when(!allHistory, till);

        ListF<String> skipReasons = Cf.list(SmsPassportErrorCode.R.toXmlName(SmsPassportErrorCode.NOCURRENT));

        return reminderSendManager.countSentAndFailedRemindersGroupedByProcessDate(
                since, Option.of(till), Option.of(SpecialClientIds.FLIGHT), Option.some(Channel.SMS), skipReasons);
    }

    public void flightTimeChanged(final String flightNumber, final int geoId, final DateTimeZone tz,
                                  final LocalDateTime plannedDateTime, final LocalDateTime actualDateTime, final ActionInfo actionInfo) {
        lockManager.withLock(
                LockManager.flightToResourceId(flightNumber, geoId, tz, plannedDateTime),
                () -> doFlightTimeChanged(flightNumber, geoId, tz, plannedDateTime, actualDateTime, actionInfo));
    }

    public FlightEventMeta getActualFlightEventMeta(Event event) {
        return getActualFlightEventMeta(event.getId().getCid(), event.getFlightMeta().toOptional());
    }

    public FlightEventMeta getActualFlightEventMeta(CallbackRequest request) {
        return getActualFlightEventMeta(request.getCid(), request.getMeta());
    }

    public FlightEventMeta getActualFlightEventMeta(String cid, Optional<FlightEventMeta> meta) {
        Check.isTrue(SpecialClientIds.isFlight(cid));

        val flightMeta =meta.get();

        val shift = flightShiftMdao.findLatestShift(flightMeta);

        Instant departureTs;

        if (shift.isPresent()) {
            departureTs = shift.get().isMigrated()
                    ? shift.get().getActualTs()
                    : DateTimeUtils.tzMigrateInstant(shift.get().getActualTs(), shift.get().getTz());

        } else {
            departureTs = flightMeta.isMigrated()
                    ? flightMeta.getDepartureTs()
                    : DateTimeUtils.tzMigrateInstant(flightMeta.getDepartureTs(), flightMeta.getDepartureCityTz());
        }
        return flightMeta.withDepartureTs(departureTs);
    }

    public ListF<Event> getFlightEventsWithActualMeta(ListF<Event> events) {
        return events.map(e -> new Event(e.getId(),
                e.getEventData().withFlightMeta(getActualFlightEventMeta(e)),
                e.getSenderName(), e.getUpdatedTs(), e.getUpdatedReqId()));
    }

    private void doFlightTimeChanged(String flightNumber, int geoId, DateTimeZone tz,
                                     LocalDateTime plannedDateTime, LocalDateTime actualDateTime, ActionInfo actionInfo) {
        val flightShift = new FlightShift(flightNumber, geoId, tz, plannedDateTime, actualDateTime);
        val freshFlightShift = flightShiftMdao.insertOrUpdate(flightShift);
        if (freshFlightShift.getSendTs().isEmpty()) {
            Instant sendTs = scheduleFlightShiftSmsSending(freshFlightShift, actionInfo);
            flightShiftMdao.updateSendTs(freshFlightShift.getId(), sendTs);
        } else {
            logger.debug("task is already scheduled for flightShift={}", freshFlightShift);
        }
    }

    private FlightRemindersInfo doCreateOrUpdateFlightReminders(PassportUid uid, String externalId,
                                                                MailReminderData mailReminderData, ListF<FlightEventMeta> flightEventMetas, ActionInfo actionInfo) {
        val oldEvents = findEvents(uid, externalId);
        val oldMailReminderData = toMailReminderDataO(oldEvents);
        if (oldMailReminderData.isPresent()) {
            mailReminderData = merge(oldMailReminderData.get(), mailReminderData);
        }

        flightEventMetas = alterFlightEventMetas(flightEventMetas, oldEvents);

        val eventIdsAndDatas =
                flightEventMetas.toTuple2List(toEventIdAndDataF(uid, externalId, mailReminderData, actionInfo));

        val currentIdxs = eventIdsAndDatas.get1().map(EventId::getIdx);
        val missingIdxs = oldEvents.map(Event::getIdx).filter(currentIdxs.containsF().notF());

        if (missingIdxs.isNotEmpty()) {
            DeleteCloudApiFlightsTask deletionTask = new DeleteCloudApiFlightsTask(
                    new DeleteCloudApiFlightsTaskParameters(uid, externalId, missingIdxs));

            bazingaTaskManager.schedule(deletionTask, Instant.now().plus(Duration.standardMinutes(10)));
        }

        val events = Tuple2List.tuple2List(eventIdsAndDatas).map(eventManager.createOrUpdateEventF(actionInfo));

        eventManager.deleteEvents(uid, SpecialClientIds.FLIGHT,
                EventsFilter.byExternalId(externalId).andByIdxNotIn(currentIdxs));

        return toFlightRemindersInfo(toMailReminderDataO(events).get(), flightEventMetas, actionInfo);
    }

    private ListF<FlightEventMeta> alterFlightEventMetas(
            ListF<FlightEventMeta> flightEventMetas, ListF<Event> oldEvents) {
        final MapF<String, FlightEventMeta> oldMetaByFlightNum =
                oldEvents.filterMap(Event.getFlightMetaF()).toMapMappingToKey(FlightEventMeta.getFlightNumberF());

        return flightEventMetas.map(newMeta -> {
            FlightEventMeta result = newMeta;
            Option<FlightEventMeta> oldMeta = oldMetaByFlightNum.getO(newMeta.getFlightNumber());
            if (oldMeta.isDefined()) {
                if (oldMeta.get().getSource().getOrNull() == FlightSource.RASP
                        && newMeta.getSource().getOrNull() != FlightSource.RASP) {
                    // we trust to RASP -> will use old meta
                    result = oldMeta.get();
                }
            }

            Option<FlightShift> latestShift = flightShiftMdao.findLatestShift(result);
            if (latestShift.isDefined()) {
                // and we trust to actual datetime from latest flight shift object
                result = result.withDepartureTs(latestShift.get().getActualTs());
            }
            return result;
        });
    }

    private FlightRemindersInfo toFlightRemindersInfo(
            MailReminderData mailReminderData, ListF<FlightEventMeta> flightEventMetas, ActionInfo actionInfo) {
        return new FlightRemindersInfo(
                mailReminderData.getUserSmsOffset(),
                mailReminderData.getPhone(),
                getAvailableOffset(flightEventMetas, actionInfo),
                mailReminderData.getAutoSmsOffset(),
                mailReminderData.getAutoEmailOffset()
        );
    }

    private FlightRemindersInfo toFlightRemindersInfo(ListF<FlightEventMeta> flightEventMetas, ActionInfo actionInfo) {
        return new FlightRemindersInfo(
                Option.empty(),
                Option.empty(),
                getAvailableOffset(flightEventMetas, actionInfo),
                Option.empty(),
                Option.empty()
        );
    }

    private Duration getAvailableOffset(ListF<FlightEventMeta> flights, ActionInfo actionInfo) {
        Option<Instant> minDepartureTs = flights.map(FlightEventMeta.getDepartureTsF())
                .filter(TimeUtils.instant.isAfterF(actionInfo.getNow())).minO();

        return minDepartureTs.isPresent() ? new Duration(minDepartureTs.get(), actionInfo.getNow()) : Duration.ZERO;
    }

    private ListF<Event> findEvents(PassportUid uid, String externalId) {
        return eventManager.findEvents(
                uid, SpecialClientIds.FLIGHT, EventsFilter.byExternalId(externalId)
        );
    }

    private MailReminderData merge(MailReminderData oldData, MailReminderData newData) {
        return new MailReminderData(
                newData.getPhone().orElse(oldData.getPhone()),
                newData.getAutoSmsOffset().orElse(oldData.getAutoSmsOffset()),
                newData.getUserSmsOffset().orElse(oldData.getUserSmsOffset()),
                newData.getEmail().orElse(oldData.getEmail()),
                newData.getAutoEmailOffset().orElse(oldData.getAutoEmailOffset())
        );
    }

    public Instant scheduleFlightShiftSmsSending(FlightShift flightShift, ActionInfo actionInfo) {
        Instant now = actionInfo.getNow();
        Instant sendTs = SendingTimeUtils.calcFlightShiftSmsSendTs(
                now, flightShift.getPlannedTs(), flightShift.getTz());
        if (sendTs.isAfter(now)) {
            logger.debug("send time is delayed (night rule), from {} till {}", now, sendTs);
        } else {
            Instant delayedSendTs = SendingTimeUtils.delayFlightShiftSmsSendTs(
                    sendTs, flightShift.getPlannedTs(), flightShift.getActualTs(), flightShift.getTz());
            if (!delayedSendTs.equals(sendTs)) {
                Check.isTrue(delayedSendTs.isAfter(sendTs));
                logger.debug("send time is delayed (more than 5 hours till flight), from {} till {}",
                        sendTs, delayedSendTs);
                sendTs = delayedSendTs;
            }
        }

        if (flightShift.getActualTs().isAfter(now)) {
            logger.debug("going to schedule task: flightShift={}, actionInfo={}, sendTs={}",
                    flightShift, actionInfo, sendTs);
            FlightShiftSendTaskParameters parameters = new FlightShiftSendTaskParameters(
                    flightShift.getId(), now, actionInfo.getRequestIdWithHostId());
            FlightShiftSendTask task = new FlightShiftSendTask(parameters);
            bazingaTaskManager.schedule(task, TaskCategory.DEFAULT, sendTs, task.priority(), true);
        } else {
            logger.debug("won't schedule task, cause new actual departure time is before now");
        }
        return sendTs;
    }

}
