package ru.yandex.reminders.logic.flight;

import org.joda.time.DateTime;
import org.joda.time.Duration;
import ru.yandex.bolts.collection.Option;
import ru.yandex.reminders.i18n.*;
import ru.yandex.reminders.logic.flight.shift.FlightShift;

public class FlightReminderSmsMessageCreator {

    public static String createReminderSmsMessage(FlightEventMeta flight) {
        return createReminderSmsMessage(flight, flight.getLanguage());
    }

    public static String createReminderSmsMessage(FlightEventMeta flight, Language lang) {
        LocalizedStringBuilder sb = new LocalizedStringBuilder();

        Option<String> departureAirportName = flight.getDepartureAirport().flatMapO(FlightItem::getName);
        if (departureAirportName.isPresent()) {
            sb.append(i18n("reminder.airport", departureAirportName.get())).append(' ');
        }
        sb.append(i18n("reminder.ticket",
                flight.getFlightNumber(),
                flight.getDepartureCity().getName(),
                flight.getArrivalCity().getName(),
                formatDateTime(flight.getDepartureDateTime(), lang)));

        return sb.build().get(lang);
    }

    public static String createPromiseSmsMessage(Duration reminderOffset, Language lang) {
        long minutes = -reminderOffset.getStandardMinutes();
        long hours = -reminderOffset.getStandardHours();
        long days = -reminderOffset.getStandardDays();

        String offsetStr = minutes < 60 ? pluralize(minutes, "promise.minutes", lang)
                : hours < 25 ? pluralize(hours, "promise.hours", lang)
                : pluralize(days, "promise.days", lang);

        return i18n("promise", offsetStr).get(lang);
    }

    public static String createFlightShiftSmsMessage(FlightEventMeta flight, FlightShift shift) {
        return createFlightShiftSmsMessage(flight, shift, flight.getLanguage());
    }

    public static String createFlightShiftSmsMessage(FlightEventMeta flight, FlightShift shift, Language lang) {
        LocalizedStringBuilder sb = new LocalizedStringBuilder();

        sb.append(i18n("shift.warn")).append(' ');
        sb.append(i18n("shift.ticket",
                flight.getFlightNumber(),
                flight.getDepartureCity().getName(),
                flight.getArrivalCity().getName(),
                formatDateTime(shift.getActualDateTime(), lang)));

        return sb.build().get(lang);
    }

    private static FormattingLocalizedString i18n(String key, Object... args) {
        return new FormattingLocalizedString(new StaticLocalizedString("flight.sms." + key), args);
    }

    private static String pluralize(long quantity, String key, Language lang) {
        return new StaticInflectingLocalizedString("flight.sms." + key).get(quantity, lang);
    }

    private static String formatDateTime(DateTime dt, Language lang) {
        return dt.toString(i18n("date").get(lang));
    }
}
