package ru.yandex.reminders.logic.flight.airport;

import lombok.val;
import org.bson.types.ObjectId;
import org.joda.time.DateTimeZone;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.commune.mongo3.bender.MongoId;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderMembersToBind;
import ru.yandex.misc.bender.annotation.BenderPart;

@Bendable
@BenderMembersToBind(value = MembersToBind.ALL_FIELDS)
public class Airport extends Named {
    private final Option<String> iata;
    private final Option<String> icao;
    private final Option<Country> country;
    private final Option<City> city;
    @BenderPart(name = "time_zone", strictName = true)
    private final Option<DateTimeZone> tz;
    // TODO: Option here is an ugly hack, correct way is to use @DefaultValue, but it doesn't work with ObjectId
    @MongoId
    private ObjectId id;

    public Airport(Option<String> name, Option<String> nameEn, Option<String> nameRu, Option<String> iata,
                   Option<String> icao, Option<Country> country, Option<City> city, Option<DateTimeZone> tz) {
        super(name, nameEn, nameRu);
        this.id = ObjectId.get();
        this.iata = iata;
        this.icao = icao;
        this.country = country;
        this.city = city;
        this.tz = tz;
    }

    public static Function1B<Airport> isIataF(final String iata) {
        return a -> a.getIata().isPresent() && a.getIata().get().equals(iata);
    }

    public static Function<Airport, Option<DateTimeZone>> getTzF() {
        return a -> a.getTz();
    }

    public static Function1B<Airport> hasTzF() {
        return getTzF().andThen(Option.isDefinedF());
    }

    public static Function<Airport, Option<String>> getCityNameEnF() {
        return a -> a.getCity().flatMapO(City.getNameEnF());
    }

    public static Function<Airport, Option<String>> getCityNameRuF() {
        return a -> a.getCity().flatMapO(City.getNameRuF());
    }

    public static Function<Airport, Option<Tuple2<String, String>>> getCityAndAirportNameRuF() {
        return a -> {
            Option<String> cityName = a.getCity().flatMapO(City.getNameRuF());
            Option<String> airportName = a.getNameRu();
            return cityName.isEmpty() || airportName.isEmpty()
                    ? Option.empty()
                    : Option.of(Tuple2.tuple(cityName.get(), airportName.get()));
        };
    }

    public static Function<Airport, Option<Tuple2<String, String>>> getCityAndAirportNameEnF() {
        return a -> {
            val cityName = a.getCity().flatMapO(City.getNameEnF());
            val airportName = a.getNameEn();
            return cityName.isEmpty() || airportName.isEmpty()
                    ? Option.empty()
                    : Option.of(Tuple2.tuple(cityName.get(), airportName.get()));
        };
    }

    public static Function<Airport, Tuple2<Option<String>, Option<String>>> getUniqueFieldsF() {
        return a -> Tuple2.tuple(a.getIata(), a.getName());
    }

    public Option<String> getIata() {
        return iata;
    }

    public Option<String> getIcao() {
        return icao;
    }

    public Option<Country> getCountry() {
        return country;
    }

    public Option<City> getCity() {
        return city;
    }

    public Option<DateTimeZone> getTz() {
        return tz;
    }

    public Option<Long> getCityGeoId() {
        return getCity().flatMapO(City::getGeoId);
    }
}
