package ru.yandex.reminders.logic.flight.airport;

import io.micrometer.core.instrument.MeterRegistry;
import org.joda.time.DateTimeZone;
import org.joda.time.LocalDateTime;
import org.springframework.beans.factory.annotation.Autowired;
import ru.yandex.bolts.collection.*;
import ru.yandex.bolts.function.Function;
import ru.yandex.commune.lockservice.ResourceId;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.reminders.logic.flight.rasp.RaspClient;
import ru.yandex.reminders.logic.update.LockManager;
import ru.yandex.reminders.util.DateTimeUtils;

import javax.annotation.PostConstruct;

public class AirportManager {
    private static final Logger logger = LoggerFactory.getLogger(AirportManager.class);
    private static final String LOCK_RESOURCE_ID = "rasp:airports";
    private static final String RESOLVE_TZ_BY_CITY_OK_METRIC = "resolve.tz.by.city.ok";
    private static final String RESOLVE_TZ_BY_CITY_AMBIGUOUS_METRIC = "resolve.tz.by.city.ambiguous";
    private static final String RESOLVE_TZ_BY_CITY_UNKNOWN_METRIC = "resolve.tz.by.city.unknown";

    @Autowired
    private LockManager lockManager;
    @Autowired
    private RaspClient raspClient;
    @Autowired
    private AirportMdao airportMdao;
    @Autowired
    private MeterRegistry registry;

    private volatile Cache cache;

    public AirportManager() {
    }

    // for tests
    public AirportManager(ListF<Airport> airports) {
        cache = new Cache(airports);
    }

    @PostConstruct
    public void init() {
        doUpdateCache();
    }

    public void importAirports() {
        lockManager.withLock(new ResourceId(LOCK_RESOURCE_ID), () -> {
            Airports airports = raspClient.getAirports();
            airportMdao.insertOrUpdate(airports.getAirports());
        });
    }

    public void updateCache() {
        lockManager.withLock(new ResourceId(LOCK_RESOURCE_ID), this::doUpdateCache);
    }

    private void doUpdateCache() {
        ListF<Airport> airports = airportMdao.findAll();
        cache = new Cache(airports);
    }

    public DateTimeZone chooseTimezoneForCityAndAirportName(String cityName, Option<String> airportName, LocalDateTime departure) {
        ListF<Airport> airports = cache.get(cityName, airportName).getOrElse(Cf.list());

        if (airports.size() > 1 && airports.count(Airport.hasTzF()) != airports.size()) {
            logger.warn("Timezone is unknown for some of cities named {}", cityName);
        }
        ListF<DateTimeZone> timezones = airports.filterMap(Airport.getTzF()).stableUnique();
        final DateTimeZone timezone;

        if (timezones.size() > 1) {
            timezone = timezones.min(DateTimeUtils.toInstantIgnoreGapF(departure).andThenNaturalComparator());
            logger.warn("More than one timezone found by city name {}, using tz={}", cityName, timezone);
            registry.counter(RESOLVE_TZ_BY_CITY_AMBIGUOUS_METRIC).increment();
        } else if (timezones.isEmpty()) {
            timezone = DateTimeZone.forOffsetHours(13);
            logger.warn("Timezone is unknown for city name {}, using tz={}", cityName, timezone);
            registry.counter(RESOLVE_TZ_BY_CITY_UNKNOWN_METRIC).increment();
        } else {
            timezone = timezones.single();
            logger.warn("Timezone is ok (single) for city name {}, using tz={}", cityName, timezone);
            registry.counter(RESOLVE_TZ_BY_CITY_OK_METRIC).increment();
        }
        return timezone;
    }

    public Option<DateTimeZone> findTimezoneByAirportCodesOrCityGeoId(Option<String> iata, Option<String> icao, Option<Integer> geoId) {
        return findAirportByCodes(iata, icao)
                .orElse(geoId.flatMapO(Cf.Integer.toLongF().andThen(cache.airportByGeoId::getO)))
                .flatMapO(Airport::getTz);
    }

    public Option<Airport> findAirportByCodes(Option<String> iata, Option<String> icao) {
        return iata.flatMapO(cache.airportByIata::getO).orElse(icao.flatMapO(cache.airportByIcao::getO));
    }

    public static class Cache {
        private final MapF<String, ListF<Airport>> airportsByCityName;
        private final MapF<Tuple2<String, String>, ListF<Airport>> airportsByCityAndAirportNames;
        private final MapF<String, Airport> airportByIata;
        private final MapF<String, Airport> airportByIcao;
        private final MapF<Long, Airport> airportByGeoId;

        public Cache(ListF<Airport> airports) {
            airportsByCityName = buildMap(
                    airports, Airport.getCityNameRuF(), Airport.getCityNameEnF(), Cf.String.toLowerCaseF());
            airportsByCityAndAirportNames = buildMap(
                    airports, Airport.getCityAndAirportNameRuF(), Airport.getCityAndAirportNameEnF(),
                    Tuple2.<String, String, String>map1F(Cf.String.toLowerCaseF())
                            .andThen(Tuple2.map2F(Cf.String.toLowerCaseF())));

            airportByIata = airports.zipWithFlatMapO(Airport::getIata).invert().toMap();
            airportByIcao = airports.zipWithFlatMapO(Airport::getIcao).invert().toMap();
            airportByGeoId = airports.zipWithFlatMapO(Airport::getCityGeoId).invert().toMap();

            logger.info("cache updated: total {} airports, byCity cache size={}, byCityAndAirport cache size()={}",
                    airports.size(), airportsByCityName.size(), airportsByCityAndAirportNames.size());
        }

        private <K> MapF<K, ListF<Airport>> buildMap(
                ListF<Airport> airports,
                Function<Airport, Option<K>> ruFieldF,
                Function<Airport, Option<K>> enFieldF,
                Function<K, K> fieldPostProcessF) {
            MapF<K, ListF<Airport>> byRuField =
                    airports.zipWithFlatMapO(ruFieldF).map2(fieldPostProcessF).groupBy2();
            MapF<K, ListF<Airport>> byEnField =
                    airports.zipWithFlatMapO(enFieldF).map2(fieldPostProcessF).groupBy2();

            return byRuField.plus(byEnField);
        }

        public Option<ListF<Airport>> get(String cityName, Option<String> airportName) {
            Option<ListF<Airport>> airportsO = Option.empty();
            if (airportName.isPresent()) {
                airportsO = airportsByCityAndAirportNames.getO(
                            Tuple2.tuple(cityName.toLowerCase(), airportName.get().toLowerCase()));
            }
            if (airportsO.isEmpty()) {
                airportsO = airportsByCityName.getO(cityName.toLowerCase());
            }
            return airportsO;
        }
    }
}
