package ru.yandex.reminders.logic.flight.airport;

import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.Filters;
import lombok.val;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.bson.types.ObjectId;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function2;
import ru.yandex.commune.mongo3.schema.IndexInfo;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.db.q.SqlOrder;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.reminders.mongodb.AbstractMdao;

public class AirportMdao extends AbstractMdao<ObjectId, Airport> {
    private static final Logger logger = LoggerFactory.getLogger(AirportMdao.class);


    public AirportMdao(MongoDatabase remindersDb) {
        super(remindersDb, "airport", Airport.class);
    }

    public static Bson toQuery(Option<String> iata, Option<String> name) {
        Function2<String, Option<String>, Bson> cons = (field, value) -> value.isPresent()
                ? Filters.eq(field, value.get())
                : Filters.exists(field, false);

        return Filters.and(cons.apply("title", name), cons.apply("iata", iata));
    }

    @Override
    public ListF<IndexInfo> getIndexes() {
        return Cf.list(new IndexInfo().key("title", 1).key("iata", 1).unique(true));
    }

    public void insertOrUpdate(ListF<Airport> airports) {
        int inserted = 0;
        int error = 0;
        for (Airport airport : airports) {
            try {
                val writeResult = collectionX.replaceOneWithoutId(
                        toQuery(airport.getIata(), airport.getName()), airport);

                if (writeResult.getMatchedCount() == 0) {
                    inserted++;
                }
            } catch (RuntimeException e) {
                logger.error("Airport {} updating failed with error: {}", airport, e);
                error++;
            }
        }
        logger.debug("processed {} airports: {} new, {} errors", airports.size(), inserted, error);
    }

    public ListF<Airport> findAll() {
        // order is required for repeatable result of choosing first element from non-unique list
        // (original FlightTimezoneResolver used stableUnique() for ambiguous cases when
        // many airports were found for the city)
        return collectionX.find(new Document(), SqlOrder.orderByColumns("iata", "title"), SqlLimits.all());
    }
}
