package ru.yandex.reminders.logic.flight.shift;

import org.bson.types.ObjectId;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDateTime;
import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.mongo3.bender.MongoId;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderMembersToBind;
import ru.yandex.misc.lang.DefaultToString;
import ru.yandex.reminders.util.task.TimeZoneMigrationTask;

@Bendable
@BenderMembersToBind(MembersToBind.ALL_FIELDS)
public class FlightShift extends DefaultToString {
    @MongoId
    private final ObjectId id;

    private final String num;  // flight number
    private final int geoId;
    private final DateTimeZone tz;
    private final Instant plannedTs;
    private final Instant actualTs;
    private final Option<Instant> sendTs;
    private final boolean latest;

    private Option<Boolean> migrate = Option.of(TimeZoneMigrationTask.MIGRATION_FLAG);

    public FlightShift(String flightNumber, int geoId, DateTimeZone tz, Instant plannedTs, Instant actualTs) {
        // intentionally null, id is always generated in mongo, because of insertOrUpdate
        this(ObjectId.get(), flightNumber, geoId, tz, plannedTs, actualTs, Option.empty(), true);
    }

    public FlightShift(String flightNumber, int geoId, DateTimeZone tz,
                       LocalDateTime plannedDateTime, LocalDateTime actualDateTime) {
        this(flightNumber, geoId, tz,
                plannedDateTime.toDateTime(tz).toInstant(), actualDateTime.toDateTime(tz).toInstant());
    }

    public FlightShift(
            ObjectId id, String flightNumber, int geoId, DateTimeZone tz,
            Instant plannedTs, Instant actualTs, Option<Instant> sendTs, boolean latest) {
        this.id = id;
        this.num = flightNumber;
        this.geoId = geoId;
        this.tz = tz;
        this.plannedTs = plannedTs;
        this.actualTs = actualTs;
        this.sendTs = sendTs;
        this.latest = latest;
    }

    public ObjectId getId() {
        return id;
    }

    public String getFlightNum() {
        return num;
    }

    public int getGeoId() {
        return geoId;
    }

    public Instant getPlannedTs() {
        return plannedTs;
    }

    public DateTimeZone getTz() {
        return tz;
    }

    public DateTime getPlannedDateTime() {
        return plannedTs.toDateTime(tz);
    }

    public Instant getActualTs() {
        return actualTs;
    }

    public DateTime getActualDateTime() {
        return actualTs.toDateTime(tz);
    }

    public Option<Instant> getSendTs() {
        return sendTs;
    }

    public boolean isLatest() {
        return latest;
    }

    public boolean isMigrated() {
        return migrate.isSome(true);
    }

    public void setNotMigrated() {
        migrate = Option.empty();
    }
}
