package ru.yandex.reminders.logic.flight.shift;

import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.Updates;
import org.apache.commons.lang.Validate;
import org.bson.conversions.Bson;
import org.bson.types.ObjectId;
import org.joda.time.Instant;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.mongo3.schema.IndexInfo;
import ru.yandex.misc.lang.Check;
import ru.yandex.reminders.logic.flight.FlightEventMeta;
import ru.yandex.reminders.mongodb.AbstractMdao;

public class FlightShiftMdao extends AbstractMdao<ObjectId, FlightShift> {

    public FlightShiftMdao(MongoDatabase remindersDb) {
        super(remindersDb, "flightShift", FlightShift.class);
    }

    private static Bson toQuery(FlightShift shift) {
        return toQuery(shift, shift.isLatest());
    }

    private static Bson toQuery(FlightShift shift, boolean latest) {
        return Filters.and(
                Filters.eq("num", shift.getFlightNum()),
                Filters.eq("geoId", shift.getGeoId()),
                Filters.eq("plannedTs", toMongoValue(shift.getPlannedTs())),
                Filters.eq("latest", latest));
    }

    private static Bson toQuery(FlightEventMeta flightEventMeta, boolean latest) {
        Check.some(flightEventMeta.getDepGeoId());
        Check.some(flightEventMeta.getPlannedDepartureTs());
        return Filters.and(
                Filters.eq("num", flightEventMeta.getFlightNumber()),
                Filters.eq("geoId", flightEventMeta.getDepGeoId().get()),
                Filters.eq("plannedTs", toMongoValue(flightEventMeta.getPlannedDepartureTs().get())),
                Filters.eq("latest", latest));
    }

    @Override
    public ListF<IndexInfo> getIndexes() {
        return Cf.list(new IndexInfo().key("num", 1).key("geoId", 1).key("plannedTs", 1).key("latest", 1));
    }

    public FlightShift insertOrUpdate(final FlightShift shift) {
        Validate.isTrue(shift.isLatest());
        ListF<FlightShift> latestShiftList = collectionX.find(toQuery(shift));
        Check.isTrue(latestShiftList.size() <= 1, "found more than 1 latest flight shift objects");
        if (latestShiftList.isNotEmpty()) {
            FlightShift latestShift = latestShiftList.first();
            if (latestShift.getActualTs().equals(shift.getActualTs())) {
                return latestShift;
            }
            updateLatest(latestShift.getId(), false);
        }
        return collectionX.findOneAndReplaceWithoutId(toQuery(shift), shift);
    }

    public Option<FlightShift> find(ObjectId id) {
        return collectionX.findById(id);
    }

    public Option<FlightShift> findLatestShift(FlightEventMeta flightEventMeta) {
        if (flightEventMeta.getDepGeoId().isEmpty() || flightEventMeta.getPlannedDepartureDateTime().isEmpty()) {
            return Option.none();
        }
        ListF<FlightShift> shifts = collectionX.find(toQuery(flightEventMeta, true));
        Check.isTrue(shifts.size() <= 1);
        return shifts.firstO();
    }

    long count() {
        return collectionX.count();
    }

    public void updateSendTs(ObjectId id, Instant sendTs) {
        collectionX.updateOneById(id, Updates.set("sendTs", toMongoValue(sendTs)));
    }

    public void updateLatest(ObjectId id, boolean latest) {
        collectionX.updateOneById(id, Updates.set("latest", latest));
    }
}
