package ru.yandex.reminders.logic.flight.shift;

import lombok.val;
import org.bson.types.ObjectId;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.impl.JobStatus;
import ru.yandex.commune.bazinga.scheduler.TaskCategory;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.thread.InterruptedRuntimeException;
import ru.yandex.misc.thread.ThreadLocalTimeoutException;
import ru.yandex.reminders.api.flight.FlightDataConverter;
import ru.yandex.reminders.logic.event.Event;
import ru.yandex.reminders.logic.event.EventMdao;
import ru.yandex.reminders.logic.event.SpecialClientIds;
import ru.yandex.reminders.logic.flight.FlightReminderSmsMessageCreator;
import ru.yandex.reminders.logic.flight.FlightSource;
import ru.yandex.reminders.logic.reminder.Channel;
import ru.yandex.reminders.logic.reminder.Reminder;
import ru.yandex.reminders.logic.reminder.SendStatus;
import ru.yandex.reminders.logic.sending.SendManagerBase;
import ru.yandex.reminders.logic.user.SettingsManager;
import ru.yandex.reminders.mongodb.BazingaOnetimeJobMdao;
import ru.yandex.reminders.worker.ReminderSendTask;
import ru.yandex.reminders.worker.ReminderSendTaskParameters;

public class FlightShiftSendManager extends SendManagerBase {
    private final DynamicProperty<Boolean> smsEnabled = new DynamicProperty<>("flightShiftSmsEnabled", false);
    @Autowired
    private SettingsManager settingsManager;
    @Autowired
    private FlightShiftMdao flightShiftMdao;
    @Autowired
    private FlightShiftSendResultMdao flightShiftSendResultMdao;
    @Autowired
    private EventMdao eventMdao;
    @Autowired
    private BazingaOnetimeJobMdao bazingaOnetimeJobMdao;
    @Autowired
    private BazingaTaskManager bazingaTaskManager;

    public FlightShiftSentInfo sendFlightShift(final ObjectId flightShiftId) {
        Option<FlightShift> shiftO = flightShiftMdao.find(flightShiftId);

        if (shiftO.isEmpty()) {
            logger.debug("Flight shift SMSes cannot be sent: the shift object not found, shiftId={}", flightShiftId);
            return FlightShiftSentInfo.failed("not-found");
        }

        FlightShift shift = shiftO.get();

        if (!shift.isLatest()) {
            logger.debug("Flight shift SMSes won't be sent: the shift object isn't latest, shiftId={}", flightShiftId);
            return FlightShiftSentInfo.failed("not-latest");
        }

        SetF<PassportUid> alreadySentUids = flightShiftSendResultMdao.findUids(shift.getId());
        ListF<Event> eventsForSending = eventMdao.findFlightEvents(
                alreadySentUids, shift.getFlightNum(), shift.getGeoId(), shift.getPlannedTs(), FlightSource.RASP);
        Integer remindersForRescheduling = eventsForSending.foldLeft(
                0, Cf.Integer.plusF().compose2(Event.getRemindersF().andThen(Cf.List.sizeF())));

        val forSendingCount = eventsForSending.size();
        val stat = new FlightShiftSendStat(
                alreadySentUids.size() + forSendingCount, forSendingCount, remindersForRescheduling);
        logger.debug("sms is already sent to {} uids, found {} uids for sending, flightShift={}",
                alreadySentUids.size(), forSendingCount, shift);

        try {
            eventsForSending.forEach((Function1V<Event>) event -> {
                try {
                    if (Thread.currentThread().isInterrupted()) {
                        throw new SendingInterruptedException("interrupted-thread-dead");
                    }
                    Option<FlightShift> shiftO1 = flightShiftMdao.find(flightShiftId);

                    if (shiftO1.isEmpty()) {
                        logger.debug(
                                "Flight shift SMSes sending interrupted: the shift object not found, shiftId={}",
                                flightShiftId);
                        throw new SendingInterruptedException("interrupted-not-found");
                    }

                    FlightShift shift1 = shiftO1.get();

                    if (!shift1.isLatest()) {
                        logger.debug(
                                "Flight shift SMSes sending interrupted: the shift object isn't latest, shiftId={}",
                                flightShiftId);
                        throw new SendingInterruptedException("interrupted-not-latest");
                    }

                    rescheduleCloudApiReminderSendJob(stat, event, shift1);
                    rescheduleReminderSendJobIfNotSent(stat, event, shift1);

                    if (!smsEnabled.get()) return;

                    SendStatus sendStatus = sendFlightShift(event, shift1, stat);
                    saveSendResult(stat, event, shift1, sendStatus);
                } catch (ThreadLocalTimeoutException e) {
                    throw new SendingInterruptedException("interrupted-timedout");
                } catch (InterruptedRuntimeException e) {
                    throw new SendingInterruptedException("interrupted-thread-dead");
                } catch (SendingInterruptedException e) {
                    throw e;
                } catch (RuntimeException e) {
                    logger.error("Error processing flight shift with id={} for event with id={}, e={}",
                            flightShiftId, event.getId(), e);
                    stat.incException();
                }
            });
        } catch (SendingInterruptedException e) {
            return FlightShiftSentInfo.failed(e.getReason());
        }

        return FlightShiftSentInfo.sent(stat);
    }

    private void rescheduleCloudApiReminderSendJob(FlightShiftSendStat stat, Event event, FlightShift shift) {
        event.getReminders().filter(Reminder.channelIsF(Channel.CLOUD_API)).forEach(reminder -> {
            val newSendTs = FlightDataConverter.calcCloudApiSendDateTime(
                    event.getFlightMeta().get(), shift.getActualDateTime(), Instant.now());

            logger.debug("Send date of reminder with id={} recalculated to {}", reminder.getId(), newSendTs);

            if (newSendTs.isPresent() && !reminder.getSendDate().equals(newSendTs.orElse(null))) {
                val task = new ReminderSendTask(new ReminderSendTaskParameters(
                        event.getId(), reminder.getId(), Instant.now(),
                        RequestIdStack.current().getOrElse("flight-shift")));

                bazingaTaskManager.schedule(
                        task, TaskCategory.DEFAULT, newSendTs.get().toInstant(), task.priority(), true);

                stat.incRemindersRescheduled();
            } else {
                stat.incRemindersNotRescheduled();
            }
        });
    }

    private void rescheduleReminderSendJobIfNotSent(final FlightShiftSendStat stat, Event event, final FlightShift shift) {
        event.getReminders().filter(Reminder.channelIsF(Channel.CLOUD_API).notF()).forEach(reminder -> {
            if (!reminder.getOffset().isDefined()) {
                logger.debug("task for reminder with id={} isn't rescheduled because of unknown offset", reminder.getId());
                stat.incRemindersNotRescheduled();

            } else {
                boolean rescheduled = false;
                DateTime newSendTs = FlightDataConverter.calcSendDateTime(
                        shift.getActualDateTime(),
                        shift.getTz(),
                        Duration.standardMinutes(reminder.getOffset().get()),
                        reminder.getChannel(), reminder.getOrigin().get());
                if (!newSendTs.equals(reminder.getSendDate())) {
                    rescheduled = bazingaOnetimeJobMdao.rescheduleReminderSendTaskByReminderIdAndStatus(
                            reminder.getId(), JobStatus.READY, newSendTs.toInstant());
                }
                if (rescheduled) {
                    logger.debug("task for reminder with id={} is rescheduled, sendTs={}", reminder.getId(), newSendTs);
                    stat.incRemindersRescheduled();
                } else {
                    logger.debug("task for reminder with id={} isn't rescheduled", reminder.getId());
                    stat.incRemindersNotRescheduled();
                }
            }
        });
    }

    private void saveSendResult(FlightShiftSendStat stat, Event event, FlightShift shift, SendStatus sendStatus) {
        logger.debug("flight shift with id={} is sent to uid={} with status={}",
                shift.getId(), event.getId(), sendStatus);
        if (sendStatus.isSent()) {
            stat.incSentOk();
            flightShiftSendResultMdao.save(FlightShiftSendResult.sent(
                    shift.getId(), event.getUid(), shift.getSendTs().get(), Channel.SMS,
                    sendStatus.asSent().getMessageId(), Instant.now()));

        } else if (sendStatus.isFailed()) {
            stat.incSentFail();
            flightShiftSendResultMdao.save(FlightShiftSendResult.failed(
                    shift.getId(), event.getUid(), shift.getSendTs().get(), Channel.SMS,
                    sendStatus.asFailed().getFailureReason(), Instant.now()));
        } else {
            stat.incSentTryAgain();
        }
    }

    private SendStatus sendFlightShift(Event event, FlightShift shift, FlightShiftSendStat stat) {
        Check.isTrue(event.isFlight());
        Check.some(event.getFlightMeta());

        if (event.getFlightMeta().get().getDepartureTs().isBefore(Instant.now())) {
            logger.error("Flight shift SMS cannot be sent because it's expired, id={}", event.getId());
            return new SendStatus.Failed("expired", Option.none());
        }

        if (settingsManager.flightRemindersDisabled(event.getUid())) {
            stat.incSentDisabled();
            logger.info("Reminder cannot be sent because user disabled flight reminders, uid={}", event.getUid());
            return new SendStatus.Failed("disabled", Option.none());
        }

        return sendSms(event, shift);
    }

    private SendStatus sendSms(Event event, FlightShift shift) {
        val message = FlightReminderSmsMessageCreator.createFlightShiftSmsMessage(event.getFlightMeta().get(), shift);
        val smsReminders = event.getReminders().filter(Reminder.channelIsF(Channel.SMS));
        return doSendSms(event.getUid(), smsReminders.flatMap(Reminder.getPhoneF()).firstO(), message, SpecialClientIds.FLIGHT);
    }

    private static class SendingInterruptedException extends RuntimeException {
        private final String reason;

        private SendingInterruptedException(String reason) {
            this.reason = reason;
        }

        public String getReason() {
            return reason;
        }
    }

}
