package ru.yandex.reminders.logic.flight.shift;

import org.joda.time.*;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.ObjectUtils;

public class SendingTimeUtils {
    private static final int NIGHT_START_HOUR = 23;
    private static final int MORNING_START_HOUR = 8;
    private static final Duration HOURS_4 = Duration.standardHours(4);
    private static final Duration HOURS_5 = Duration.standardHours(5);
    private static final Duration MINS_30 = Duration.standardMinutes(30);
    private static final LocalTime MORNING = new LocalTime(MORNING_START_HOUR, 0);
    private static final LocalTime NIGHT = new LocalTime(NIGHT_START_HOUR, 0);

    public static Instant delayFlightShiftSmsSendTs(
            Instant sendTs, Instant plannedDepTs, Instant actualDepTs, DateTimeZone tz) {
        Instant minTs = ObjectUtils.min(plannedDepTs, actualDepTs);
        if (sendTs.isAfter(minTs)) {
            return sendTs;
        }
        Duration durationToMin = new Duration(sendTs, minTs);
        if (durationToMin.isShorterThan(HOURS_5)) {
            return sendTs;
        }

        Instant delayedSendTs = sendTs.plus(MINS_30);
        return isDay(sendTs.toDateTime(tz).toLocalTime()) && !isDay(delayedSendTs.toDateTime(tz).toLocalTime())
                ? sendTs
                : delayedSendTs;
    }

    // algorithm for https://jira.yandex-team.ru/browse/DARIA-28984?focusedCommentId=4515924
    public static Instant calcFlightShiftSmsSendTs(Instant now, Instant plannedDepTs, DateTimeZone tz) {
        if (now.isAfter(plannedDepTs)) {
            return now;
        }
        LocalTime nowLocalTime = now.toDateTime(tz).toLocalTime();
        if (isDay(nowLocalTime)) {
            return now;
        }
        Instant nextMorning = nextMorning(now, tz);
        return plannedDepTs.isBefore(nextMorning.plus(HOURS_4)) ? now : nextMorning;
    }

    public static LocalDateTime calcFlightReminderAutoSmsSendTs(LocalDateTime sendTs) {
        // DARIA-24055
        LocalTime sendLocalTime = sendTs.toLocalTime();
        if (sendLocalTime.isAfter(NIGHT)) {
            return sendTs.plusDays(1).withFields(MORNING);
        } else if (sendLocalTime.isBefore(MORNING)) {
            return sendTs.withFields(MORNING);
        }
        return sendTs;
    }

    static boolean isDay(LocalTime t) {
        return isBetweenHours(t, MORNING_START_HOUR, NIGHT_START_HOUR);
    }

    static boolean isBetweenHours(LocalTime t, int start, int end) {
        Check.lt(start, end);
        int hourOfDay = t.getHourOfDay();
        return hourOfDay >= start && hourOfDay < end;
    }

    static Instant nextMorning(Instant t, DateTimeZone tz) {
        DateTime dateTime = t.toDateTime(tz);
        if (dateTime.toLocalTime().isAfter(MORNING)) {
            dateTime = dateTime.plusDays(1);
        }
        return dateTime.withFields(MORNING).toInstant();
    }

    static Instant thisNight(Instant t, DateTimeZone tz) {
        LocalTime localTime = t.toDateTime(tz).toLocalTime();
        return t.toDateTime(tz).plusDays(localTime.isBefore(MORNING) ? -1 : 0).withFields(NIGHT).toInstant();
    }
}
