package ru.yandex.reminders.logic.panel;

import lombok.val;
import org.joda.time.LocalDateTime;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.json.JsonNull;
import ru.yandex.commune.json.JsonNumber;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonString;
import ru.yandex.reminders.api.DateTimeConverters;
import ru.yandex.reminders.logic.event.Event;
import ru.yandex.reminders.logic.event.EventId;

import java.util.Optional;

import static ru.yandex.reminders.util.JsonObjectUtils.getObjectFieldO;
import static ru.yandex.reminders.util.JsonObjectUtils.getStringFieldValueO;

public class HotelConverter {

    public static Hotel convertHotelReservation(EventId id, JsonObject jsonData) {
        String missMessage = " is missing for " + id;
        JsonObject data = getObjectFieldO(jsonData, "data").orElseThrow(() -> new RuntimeException("data " + missMessage));

        String domain = getStringFieldValueO(data, "domain").orElseThrow(() -> new RuntimeException("domain " + missMessage));
        String hotel = getStringFieldValueO(data, "hotel").orElseThrow(() -> new RuntimeException("hotel " + missMessage));
        String reservationNumber = getStringFieldValueO(data, "reservation_number").
                orElseThrow(() -> new RuntimeException("reservation_number " + missMessage));

        LocalDateTime checkinTime = getDateFieldValue(data, "check-inn_date", missMessage);
        LocalDateTime checkoutTime = getDateFieldValue(data, "check-out_date", missMessage);

        val address = getStringFieldValueO(data, "address");

        val cityName = getStringFieldValueO(data, "city");
        val cityGeoId = getStringOrNumberFieldIntegerValueO(data, "city_geoid");

        val countryName = getStringFieldValueO(data, "country");
        val countryGeoId = getStringOrNumberFieldIntegerValueO(data, "country_geoid");

        val cancellationDueTime = getStringFieldValueO(data, "cancellation_date")
                .map(DateTimeConverters.hotelDateParser);

        val numNights = getStringOrNumberFieldIntegerValueO(data, "number_of_nights");
        val numPersons = getStringOrNumberFieldIntegerValueO(data, "people");
        val price = getStringFieldValueO(data, "price");
        val editLink = getStringFieldValueO(data, "link");

        return new Hotel(
                domain, reservationNumber, hotel,
                checkinTime, checkoutTime, cancellationDueTime,
                address, cityName, cityGeoId, countryName, countryGeoId,
                numNights, numPersons, price, editLink);
    }

    private static LocalDateTime getDateFieldValue(JsonObject object, String fieldName, String missMess) {
        val dateStr = getStringFieldValueO(object, fieldName).orElseThrow(() -> new RuntimeException(fieldName + missMess));

        return DateTimeConverters.hotelDateParser.apply(dateStr);
    }

    private static Optional<Integer> getStringOrNumberFieldIntegerValueO(JsonObject object, String fieldName) {
        val valueO = object.getO(fieldName);
        if (valueO.isEmpty() || valueO.get().getClass().isAssignableFrom(JsonNull.class)) {
            return Optional.empty();
        }
        val value = valueO.get();

        if (value.getClass().isAssignableFrom(JsonNumber.class)) {
            return Optional.of(((JsonNumber) value).getValue().intValue());
        }
        if (value.getClass().isAssignableFrom(JsonString.class)) {
            return Cf.Integer.parseSafe(((JsonString) value).getValue()).toOptional();
        }
        throw new AssertionError("Field " + fieldName + " is of unexpected type " + value.getClass().getSimpleName());
    }
}
