package ru.yandex.reminders.logic.panel;

import java.io.UnsupportedEncodingException;
import java.util.Optional;

import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.misc.lang.Validate;
import ru.yandex.reminders.logic.event.EventId;
import ru.yandex.reminders.logic.event.SpecialClientIds;
import ru.yandex.reminders.logic.flight.FlightEventMeta;
import ru.yandex.reminders.logic.reminder.SendStatus;

@Slf4j
@AllArgsConstructor
public class PanelManager {
    private final PanelPushClient panelPushClient;

    public SendStatus sendFlight(EventId id, Optional<FlightEventMeta> meta) {
        return send(id, "flight-reminder", () -> {
            Validate.present(meta, "Not a flight event");
            return new Flight(meta.get());
        });
    }

    public SendStatus sendHotel(EventId id, JsonObject data) {
        return send(id, "hotel-reminder", () -> {
            Validate.isTrue(SpecialClientIds.isHotel(id.getCid()), "Not a hotel event");
            return HotelConverter.convertHotelReservation(id, data);
        });
    }

    private SendStatus send(EventId id, String eventType, Function0<?> bendableF) {
        String message;
        try {
            message = new String(PanelBender.mapper.serializeJson(bendableF.apply()), "utf-8");

        } catch (RuntimeException | UnsupportedEncodingException e) {
            log.error("Failed to convert event to panel push message", e);

            return new SendStatus.Failed(e.getMessage(), Option.empty());
        }
        panelPushClient.push(id.getUid(), eventType, message);

        return new SendStatus.Sent(Option.empty());
    }
}
