package ru.yandex.reminders.logic.reminder;

import org.bson.types.ObjectId;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderFlatten;
import ru.yandex.misc.bender.annotation.BenderMembersToBind;
import ru.yandex.misc.email.Email;
import ru.yandex.reminders.api.reminder.ChannelsData;
import ru.yandex.reminders.mongodb.CompressibleString;

@Bendable
@BenderMembersToBind(MembersToBind.ALL_FIELDS)
public class Reminder {
    private final ObjectId id;

    private final Instant sendTs;
    private final String sendTz;

    private final Option<Integer> offset;
    private final Option<Origin> origin; // some() only for flight reminder, none() for other reminder types

    private final Channel channel;

    private final Option<PhoneNumber> phone;
    private final Option<CompressibleString> text;

    private final Option<String> from; // CAL-6586
    private final Option<Email> email;
    private final Option<CompressibleString> subject;
    private final Option<CompressibleString> bodyText;
    private final Option<CompressibleString> bodyHtml;

    private final Option<String> url;
    private final Option<CompressibleString> message;
    @BenderFlatten
    private final ChannelsData.Sup.SupExtraFields supExtraFields;

    public static Reminder sms(
            DateTime sendDate, Option<Integer> offset, Option<Origin> origin,
            Option<PhoneNumber> phone, Option<String> text)
    {
        return new Reminder(new ObjectId(), sendDate, offset, origin, Channel.SMS,
                phone, text, Option.<String>none(), Option.<Email>none(), Option.<String>none(),
                Option.<String>none(), Option.<String>none(), Option.<String>none(), Option.<String>none(),
                new ChannelsData.Sup.SupExtraFields());
    }

    public static Reminder email(
            DateTime sendDate, Option<Integer> offset, Option<Origin> origin,
            Option<String> from, Option<Email> email, Option<String> subject,
            Option<String> bodyText, Option<String> bodyHtml)
    {
        return new Reminder(new ObjectId(), sendDate, offset, origin, Channel.EMAIL,
                Option.<PhoneNumber>none(), Option.<String>none(), from, email, subject, bodyText, bodyHtml,
                Option.<String>none(), Option.<String>none(), new ChannelsData.Sup.SupExtraFields());
    }

    public static Reminder panel(DateTime sendDate, Option<String> url, Option<String> subject, Option<String> message)
    {
        return new Reminder(new ObjectId(), sendDate, Option.<Integer>none(), Option.<Origin>none(), Channel.PANEL,
                Option.<PhoneNumber>none(), Option.<String>none(),
                Option.<String>none(), Option.<Email>none(), subject, Option.<String>none(), Option.<String>none(),
                url, message, new ChannelsData.Sup.SupExtraFields());
    }

    public static Reminder callback(DateTime sendDate, String url)
    {
        return new Reminder(new ObjectId(), sendDate, Option.<Integer>none(), Option.<Origin>none(), Channel.CALLBACK,
                Option.<PhoneNumber>none(), Option.<String>none(),
                Option.<String>none(), Option.<Email>none(), Option.<String>none(),
                Option.<String>none(), Option.<String>none(), Option.some(url), Option.<String>none(), new ChannelsData.Sup.SupExtraFields());
    }

    public static Reminder cloudApi(DateTime sendDate) {
        return new Reminder(new ObjectId(), sendDate, Option.none(), Option.none(), Channel.CLOUD_API,
                Option.none(), Option.none(),
                Option.none(), Option.none(), Option.none(),
                Option.none(), Option.none(), Option.none(), Option.none(), new ChannelsData.Sup.SupExtraFields());
    }

    public static Reminder sup(DateTime sendDate, Option<String> title, Option<String> body, Option<String> uri,
                               ChannelsData.Sup.SupExtraFields supExtraFields) {
        return new Reminder(
                new ObjectId(), sendDate, Option.empty(), Option.empty(), Channel.SUP,
                Option.empty(), Option.empty(), Option.empty(), Option.empty(), title, Option.empty(),
                Option.empty(), uri, body, supExtraFields);
    }

    public static Reminder xiva(DateTime sendDate, Origin origin, String uri) {
        return new Reminder(
                new ObjectId(), sendDate, Option.empty(), Option.of(origin), Channel.XIVA,
                Option.empty(), Option.empty(), Option.empty(), Option.empty(), Option.empty(), Option.empty(),
                Option.empty(), Option.of(uri), Option.empty(), new ChannelsData.Sup.SupExtraFields());
    }

    Reminder(
            ObjectId id, DateTime sendDate, Option<Integer> offset, Option<Origin> origin, Channel channel,
            Option<PhoneNumber> phone, Option<String> text,
            Option<String> from, Option<Email> email, Option<String> subject,
            Option<String> bodyText, Option<String> bodyHtml,
            Option<String> url, Option<String> message, ChannelsData.Sup.SupExtraFields extraFields)
    {
        this.id = id;
        this.sendTs = sendDate.toInstant();
        this.sendTz = sendDate.getZone().getID();
        this.offset = offset;
        this.origin = origin;
        this.channel = channel;
        this.phone = phone;
        this.text = text.map(CompressibleString.consF());
        this.from = from;
        this.email = email;
        this.subject = subject.map(CompressibleString.consF());
        this.bodyText = bodyText.map(CompressibleString.consF());
        this.bodyHtml = bodyHtml.map(CompressibleString.consF());
        this.url = url;
        this.message = message.map(CompressibleString.consF());
        this.supExtraFields = extraFields;
    }

    public Reminder withId(ObjectId id) {
        return new Reminder(
                id, getSendDate(), getOffset(), getOrigin(), getChannel(),
                getPhone(), getText(), getFrom(), getEmail(), getSubject(), getBodyText(), getBodyHtml(),
                getUrl(), getMessage(), getSupExtraFields());
    }

    public Reminder withNoneSmsText() {
        return text.isEmpty() ? this : new Reminder(
                id, getSendDate(), getOffset(), getOrigin(), getChannel(),
                getPhone(), Option.<String>none(),
                getFrom(), getEmail(), getSubject(), getBodyText(), getBodyHtml(), getUrl(),
                getMessage(), getSupExtraFields());
    }

    public Reminder withNoneEmailParts() {
        return subject.isEmpty() && bodyText.isEmpty() && bodyHtml.isEmpty() ? this : new Reminder(
                id, getSendDate(), getOffset(), getOrigin(), getChannel(),
                getPhone(), getText(),
                getFrom(), getEmail(), Option.<String>none(), Option.<String>none(), Option.<String>none(),
                getUrl(), getMessage(), getSupExtraFields());
    }

    public boolean isSms() {
        return channel == Channel.SMS;
    }

    public boolean isEmail() {
        return channel == Channel.EMAIL;
    }

    public boolean isPanel() {
        return channel == Channel.PANEL;
    }

    public boolean isCallback() {
        return channel == Channel.CALLBACK;
    }

    public boolean isSup() {
        return channel == Channel.SUP;
    }

    public boolean isXiva() {
        return channel == Channel.XIVA;
    }

    public ObjectId getId() {
        return id;
    }

    public Instant getSendTs() {
        return sendTs;
    }

    public DateTime getSendDate() {
        return sendTs.toDateTime(getSendTz());
    }

    public Option<DateTime> getEventDate() {
        return offset.map(o -> getSendDate().minusMinutes(o));
    }

    public Option<Instant> getEventTs() {
        return getEventDate().map(DateTime::toInstant);
    }

    public DateTimeZone getSendTz() {
        return DateTimeZone.forID(sendTz);
    }

    public Option<Integer> getOffset() {
        return offset;
    }

    public Option<Origin> getOrigin() {
        return origin;
    }

    public Channel getChannel() {
        return channel;
    }

    public Option<PhoneNumber> getPhone() {
        return phone;
    }

    public Option<String> getText() {
        return text.map(CompressibleString.getStringF());
    }

    public Option<String> getFrom() {
        return from;
    }

    public Option<Email> getEmail() {
        return email;
    }

    public Option<String> getSubject() {
        return subject.map(CompressibleString.getStringF());
    }

    public Option<String> getBodyText() {
        return bodyText.map(CompressibleString.getStringF());
    }

    public Option<String> getBodyHtml() {
        return bodyHtml.map(CompressibleString.getStringF());
    }

    public Option<String> getUrl() {
        return url;
    }

    public ChannelsData.Sup.SupExtraFields getSupExtraFields() {
        return supExtraFields;
    }

    public Option<String> getMessage() {
        return message.map(CompressibleString.getStringF());
    }

    public Reminder withSendTs(Instant ts) {
        return new Reminder(
                getId(), ts.toDateTime(getSendTz()),
                getOffset(), getOrigin(), getChannel(), getPhone(), getText(),
                getFrom(), getEmail(), getSubject(), getBodyText(), getBodyHtml(), getUrl(),
                getMessage(), getSupExtraFields());
    }

    public static Function<Reminder, Reminder> withIdF(final Function<Reminder, ObjectId> idF) {
        return r -> r.withId(idF.apply(r));
    }

    public static Function<Reminder, ObjectId> getIdF() {
        return r -> r.getId();
    }

    public static Function<Reminder, Instant> getSendTsF() {
        return r -> r.getSendTs();
    }

    public static Function<Reminder, DateTime> getSendDateF() {
        return r -> r.getSendDate();
    }

    public static Function<Reminder, Option<Integer>> getOffsetF() {
        return r -> r.getOffset();
    }

    public static Function<Reminder, Option<Origin>> getOriginF() {
        return r -> r.getOrigin();
    }

    public static Function1B<Reminder> originIsF(Origin origin) {
        return getOriginF().andThenEquals(Option.some(origin));
    }

    public static Function<Reminder, Channel> getChannelF() {
        return r -> r.getChannel();
    }

    public static Function1B<Reminder> channelIsF(Channel channel) {
        return getChannelF().andThenEquals(channel);
    }
    public static Function<Reminder, Option<PhoneNumber>> getPhoneF() {
        return r -> r.getPhone();
    }

    public static Function<Reminder, Option<String>> getTextF() {
        return r -> r.getText();
    }

    public static Function<Reminder, Option<Email>> getEmailF() {
        return r -> r.getEmail();
    }

    public static Function<Reminder, Option<String>> getSubjectF() {
        return r -> r.getSubject();
    }

    public static Function<Reminder, Option<String>> getBodyTextF() {
        return r -> r.getBodyText();
    }

    public static Function<Reminder, Option<String>> getBodyHtmlF() {
        return r -> r.getBodyHtml();
    }

}
