package ru.yandex.reminders.logic.reminder;

import io.micrometer.core.instrument.MeterRegistry;
import lombok.val;
import org.bson.types.ObjectId;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.mail.MailMessage;
import ru.yandex.misc.lang.Validate;
import ru.yandex.reminders.log.TskvReminderSendLogger;
import ru.yandex.reminders.logic.callback.CallbackManager;
import ru.yandex.reminders.logic.callmeback.in.CallbackRequest;
import ru.yandex.reminders.logic.cloudApi.CloudApiClient;
import ru.yandex.reminders.logic.event.*;
import ru.yandex.reminders.logic.flight.FlightEventMeta;
import ru.yandex.reminders.logic.flight.FlightReminderMailMessageCreator;
import ru.yandex.reminders.logic.flight.FlightReminderManager;
import ru.yandex.reminders.logic.flight.FlightReminderSmsMessageCreator;
import ru.yandex.reminders.logic.panel.PanelManager;
import ru.yandex.reminders.logic.sending.SendManagerBase;
import ru.yandex.reminders.logic.sup.SupPushManager;
import ru.yandex.reminders.logic.tv.TvReminderManager;
import ru.yandex.reminders.logic.user.SettingsManager;

import java.util.stream.Stream;

public class ReminderSendManager extends SendManagerBase {
    private static final Duration EXPIRE_INTERVAL = Duration.standardDays(1);
    private static final String SEND_REMINDERS_METRIC = "reminders.send";

    @Autowired
    private SettingsManager settingsManager;
    @Autowired
    private EventManager eventManager;
    @Autowired
    private SendResultMdao sendResultMdao;
    @Autowired
    private FlightReminderManager flightReminderManager;
    @Autowired
    private TvReminderManager tvReminderManager;
    @Autowired
    private PanelManager panelManager;
    @Autowired
    private CallbackManager callbackManager;
    @Autowired
    private CloudApiClient cloudApiClient;
    @Autowired
    private SupPushManager supPushManager;
    @Autowired
    private MeterRegistry registry;

    private static String getClientId(String cid) {
        if (!SpecialClientIds.isSpecial(cid)) {
            return "OTHERS";
        }
        return cid;
    }

    private static String getChannel(Reminder reminder) {
        return reminder.getChannel().toString().toLowerCase();
    }

    public SendStatus sendReminder(EventId eventId, ObjectId reminderId) {
        val eventO = eventManager.findEvent(eventId);
        val reminderO = eventO.flatMap(Event.getRemindersF())
                .find(Reminder.getIdF().andThenEquals(reminderId));

        // we intentionally don't count not-found case by monica
        // cause it is ok that scheduled reminder not found after event was updated
        if (reminderO.isEmpty()) {
            logger.debug("Reminder cannot be sent because it's not found, id={}", reminderId);
            return new SendStatus.Failed("not-found", Option.empty());
        }

        val clientId = getClientId(eventId.getCid());
        val channel = getChannel(reminderO.get());

        try {
            val result = sendReminder(eventO.get(), reminderO.get());

            TskvReminderSendLogger.log(eventO.get(), reminderO.get(), result);

            if (result.isSent()) {
                sendResultMdao.saveSendResult(SendResult.sent(
                        eventId, reminderId, reminderO.get().getSendTs(), reminderO.get().getChannel(),
                        result.asSent().getMessageId(), Instant.now()));
                updateSendMetrics(clientId, channel, "success");
            } else if (result.isFailed()) {
                sendResultMdao.saveSendResult(SendResult.failed(
                        eventId, reminderId, reminderO.get().getSendTs(), reminderO.get().getChannel(),
                        result.asFailed().getFailureReason(), Instant.now()));
                updateSendMetrics(clientId, channel, "failed");
            } else if (result.isTryAgain()) {
                updateSendMetrics(clientId, channel, "failed.attempt");
            } else {
                throw new RuntimeException("Unknown status class=" + result.getClass().getName());
            }
            return result;
        } catch (RuntimeException e) {
            updateSendMetrics(clientId, channel, "failed.exception");
            throw e;
        }
    }

    private void updateSendMetrics(String clientId, String channel, String status) {
        val normalizeChannel = channel.replace("-", "_").toLowerCase();
        val normalizeClientId = clientId.replace("-", "_").toLowerCase();
        Stream.of(
                status,
                normalizeChannel,
                normalizeClientId,
                normalizeClientId + "." + normalizeChannel,
                normalizeClientId + "." + status,
                normalizeChannel + "." + status,
                normalizeChannel + "." + status,
                normalizeClientId + "." + normalizeChannel + "." + status)
        .map(metricName -> SEND_REMINDERS_METRIC + "." + metricName)
        .forEach(name -> registry.counter(name).increment());
    }

    public ListF<SendDailyStat> countSentAndFailedRemindersGroupedByProcessDate(
            Option<LocalDate> since, Option<LocalDate> till,
            Option<String> clientId, Option<Channel> channel, ListF<String> skipFailReasons) {
        return sendResultMdao.countSentAndFailedRemindersGroupedByProcessDate(since, till, clientId, channel, skipFailReasons);
    }

    public SendStatus sendReminder(CallbackRequest request) {
        val clientId = getClientId(request.getCid());
        val channel = getChannel(request.getReminder());
        try {
            val result = sendReminderNoLog(request);
            return result;
        } catch (RuntimeException e) {
            // TODO  А надо ли фиксировать метрику здесь, или пусть новый callmeback это делает
            updateSendMetrics(clientId, channel, "failed.exception");
            throw e;
        }
    }

    private SendStatus sendReminderNoLog(CallbackRequest request) {
        val reminder = request.getReminder();
        if (reminder.getSendDate().isBefore(Instant.now().minus(EXPIRE_INTERVAL))) {
            logger.error("Reminder cannot be sent because it's expired, id={}", reminder.getId());
            return new SendStatus.Failed("expired", Option.empty());
        }

        val uid = request.getUid();
        val cid = request.getCid();

        if (SpecialClientIds.isFlight(cid)) {
            logger.info("Reminder cannot be sent because flight reminders disabled");
            return new SendStatus.Failed("disabled", Option.empty());
        }

        if (SpecialClientIds.isFlight(cid) && settingsManager.flightRemindersDisabled(uid)) {
            logger.info("Reminder cannot be sent because user disabled flight reminders, uid={}", uid);
            return new SendStatus.Failed("disabled", Option.empty());
        }


        if (SpecialClientIds.isTv(cid)) { // CAL-6982
            boolean canSend;
            try {
                canSend = tvReminderManager.askTvCanReminderBeSent(reminder, request.getId());
            } catch (RuntimeException e) {
                if (reminder.getSendDate().plusMinutes(2).isAfter(Instant.now())) {
                    throw e;
                }
                logger.info("Do not know what Tv think about sending," +
                        " but going to send reminder anyway, id={}, error: {}", reminder.getId(), e);
                canSend = true;
            }
            if (!canSend) {
                logger.info("Reminder cannot be sent because Tv does not let, id={}", reminder.getId());
                return new SendStatus.Failed("disabled", Option.empty());
            }
        }

        switch (reminder.getChannel()) {
            case SMS:
                return sendSms(request);
            case EMAIL:
                return sendEmail(request);
            case PANEL:
                return sendPanelNote(request);
            case CALLBACK:
                return callbackManager.invoke(request.getId(), reminder);
            case CLOUD_API:
                Validate.isTrue(SpecialClientIds.isFlight(cid), "Expected flight event");

                val id = request.getId().getExtId() + "." + request.getId().getIdx();
                cloudApiClient.sendFlight(uid, id, flightReminderManager.getActualFlightEventMeta(request));
                return new SendStatus.Sent(Option.empty());
            case SUP:
                return supPushManager.pushSup(request);
            case XIVA:
                return supPushManager.pushXiva(request.getId(), reminder);
            default:
                throw new IllegalArgumentException("Unexpected reminder channel");
        }
    }

    private SendStatus sendReminder(Event event, Reminder reminder) {
        return sendReminderNoLog(CallbackRequest.create(event, reminder));
    }

    private SendStatus sendSms(CallbackRequest request) {
        String message;

        val cid = request.getCid();
        val reminder = request.getReminder();

        if (SpecialClientIds.isFlight(cid)) {
            message = FlightReminderSmsMessageCreator.createReminderSmsMessage(getActualFlightEventMeta(request));
        } else if (SpecialClientIds.isTv(cid)) {
            message = reminder.getText().get().replace("${advanceMinutes}", "" + -reminder.getOffset().get());
        } else {
            message = EventReminderMessageCreator.createSms(request);
        }
        return doSendSms(request.getUid(), reminder.getPhone(), message, cid);
    }

    private SendStatus sendEmail(CallbackRequest request) {
        val reminder = request.getReminder();
        val cid = request.getCid();
        MailMessage message;
        if (SpecialClientIds.isFlight(cid)) {
            message = FlightReminderMailMessageCreator.create(
                    reminder.getEmail().get(), getActualFlightEventMeta(request));
        } else {
            message = EventReminderMessageCreator.createMail(request);
        }

        return doSendEmail(request.getUid(), reminder.getEmail(), message);
    }

    private SendStatus sendPanelNote(CallbackRequest request) {
        val cid = request.getCid();
        if (SpecialClientIds.isFlight(cid)) {
            return panelManager.sendFlight(request.getId(), request.getMeta());
        } else if (SpecialClientIds.isHotel(cid)) {
            return panelManager.sendHotel(request.getId(), request.getData().getData().get());
        } else {
            return new SendStatus.Failed("panel note is not supported for clientId=" + cid, Option.empty());
        }
    }

    private FlightEventMeta getActualFlightEventMeta(CallbackRequest request) {
        return flightReminderManager.getActualFlightEventMeta(request);
    }
}
