package ru.yandex.reminders.logic.reminder;

import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.Accumulators;
import com.mongodb.client.model.Aggregates;
import com.mongodb.client.model.Filters;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.bson.types.ObjectId;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.mongo3.FiltersX;
import ru.yandex.commune.mongo3.schema.IndexInfo;
import ru.yandex.reminders.mongodb.AbstractMdao;

/**
 * @author dbrylev
 */
public class SendResultMdao extends AbstractMdao<ObjectId, SendResult> {

    public SendResultMdao(MongoDatabase remindersDb) {
        super(remindersDb, "sendResult", SendResult.class);
    }

    @Override
    public ListF<IndexInfo> getIndexes() {
        return Cf.list(new IndexInfo().key("cid", 1).key("processTs", 1).key("channel", 1));
    }

    public void saveSendResult(SendResult result) {
        collectionX.tryInsert(result);
    }

    public void saveSendResults(ListF<SendResult> results) {
        collectionX.insertMany(results);
    }

    public ListF<SendDailyStat> countSentAndFailedRemindersGroupedByProcessDate(
            Option<LocalDate> since, Option<LocalDate> till,
            Option<String> clientId, Option<Channel> channel, ListF<String> skipFailReasons)
    {
        ListF<Bson> filters = Cf.arrayList();

        since.forEach(dt -> filters.add(Filters.gte("processTs", toMongoValue(dt))));
        till.forEach(dt -> filters.add(Filters.lt("processTs", toMongoValue(dt.plusDays(1)))));

        clientId.forEach(cid -> filters.add(Filters.eq("cid", cid)));
        channel.forEach(ch -> filters.add(Filters.eq("channel", ch.value())));

        skipFailReasons.forEach(rs -> filters.add(Filters.nin("fail", skipFailReasons)));

        Bson match = Aggregates.match(filters.isNotEmpty() ? Filters.and(filters) : new Document());

        Bson extractYear = new Document("$substr", Cf.list(Cf.map("$year", "$processTs"), 0, 4));
        Bson extractMonth = new Document("$substr", Cf.list(Cf.map("$month", "$processTs"), 0, 2));
        Bson extractDay = new Document("$substr", Cf.list(Cf.map("$dayOfMonth", "$processTs"), 0, 2));

        Bson isSent = new Document("$eq", Cf.list("$status", SendResultStatus.SENT.value()));
        Bson isFailed = new Document("$eq", Cf.list("$status", SendResultStatus.FAILED.value()));

        Bson aggregate = Aggregates.group(
                new Document("$concat", Cf.list(extractYear, "-", extractMonth, "-", extractDay)),
                Accumulators.sum("sent", new Document("$cond", Cf.list(isSent, 1, 0))),
                Accumulators.sum("failed", new Document("$cond", Cf.list(isFailed, 1, 0))));

        return collectionX.aggregate(match, aggregate).map(doc -> new SendDailyStat(
                LocalDate.parse(doc.get(FiltersX.ID_FIELD).asString().getValue()),
                doc.get("sent").asNumber().intValue(),
                doc.get("failed").asNumber().intValue()));
    }

    public ListF<SendResult> find(Bson query) {
        return collectionX.find(query);
    }
}
