package ru.yandex.reminders.logic.reminder;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.misc.lang.DefaultToString;
import ru.yandex.reminders.log.TskvFields;
import ru.yandex.reminders.log.TskvLogDataSource;

public abstract class SendStatus extends DefaultToString implements TskvLogDataSource {

    public static class Sent extends SendStatus {
        private final Option<String> messageId;

        public Sent(Option<String> address) {
            super(address);
            this.messageId = Option.empty();
        }

        public Sent(String messageId, Option<String> address) {
            super(address);
            this.messageId = Option.of(messageId);
        }

        public Option<String> getMessageId() {
            return messageId;
        }

        @Override
        public Tuple2List<String, String> tskvLogData() {
            return super.tskvLogData()
                    .plus1(Tuple2.tuple(TskvFields.STATUS, "succeeded"))
                    .plus(messageId.map(id -> Tuple2.tuple(TskvFields.MESSAGE_ID, id)));
        }
    }

    public static class Failed extends SendStatus {
        private final String failureReason;

        public Failed(String failureReason, Option<String> address) {
            super(address);
            this.failureReason = failureReason;
        }

        public String getFailureReason() {
            return failureReason;
        }

        @Override
        public Tuple2List<String, String> tskvLogData() {
            return super.tskvLogData()
                    .plus(Tuple2List.fromPairs(TskvFields.STATUS, "failed", TskvFields.ERROR, failureReason));
        }
    }

    public static class TryAgain extends SendStatus {
        private final String message;

        public TryAgain(String message, Option<String> address) {
            super(address);
            this.message = message;
        }

        public String getMessage() {
            return message;
        }

        @Override
        public Tuple2List<String, String> tskvLogData() {
            return super.tskvLogData()
                    .plus(Tuple2List.fromPairs(TskvFields.STATUS, "try_again", TskvFields.ERROR, message));
        }
    }

    private final Option<String> address;

    public SendStatus(Option<String> address) {
        this.address = address;
    }

    @Override
    public Tuple2List<String, String> tskvLogData() {
        return Tuple2List.tuple2List(address.map(addr -> Tuple2.tuple(TskvFields.ADDRESS, addr)));
    }

    public boolean isSent() {
        return this instanceof Sent;
    }

    public boolean isFailed() {
        return this instanceof Failed;
    }

    public boolean isTryAgain() {
        return this instanceof TryAgain;
    }

    public Sent asSent() {
        return (Sent) this;
    }

    public Failed asFailed() {
        return (Failed) this;
    }

    public TryAgain asTryAgain() {
        return (TryAgain) this;
    }
}
