package ru.yandex.reminders.logic.sending;

import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.mail.HeaderNames;
import ru.yandex.commune.mail.MailAddress;
import ru.yandex.commune.mail.MailMessage;
import ru.yandex.commune.mail.send.MailSender;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.passport.blackbox2.protocol.request.params.EmailsParameterValue;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;
import ru.yandex.inside.passport.sms.SmsPassportErrorCode;
import ru.yandex.inside.passport.sms.SmsPassportException;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.net.LocalhostUtils;
import ru.yandex.reminders.logic.reminder.PhoneNumber;
import ru.yandex.reminders.logic.reminder.SendStatus;
import ru.yandex.reminders.util.MailUtils;

public class SendManagerBase {
    protected final Logger logger = LoggerFactory.getLogger(getClass());

    @Autowired
    private SmsSender smsSender;
    @Autowired
    private MailSender mailSender;
    @Autowired
    private Blackbox2 blackbox;

    protected SendStatus doSendSms(PassportUid uid, Option<PhoneNumber> phone, String message, String cid) {
        String address = phone.map(PhoneNumber::getPhone).getOrElse("current");
        try {
            return new SendStatus.Sent(smsSender.sendToUser(uid, phone, message, cid), Option.of(address));

        } catch (SmsPassportException ex) {
            logger.error("Passport error occurred while sending sms " + ex);

            if (Cf.list(SmsPassportErrorCode.INTERROR, SmsPassportErrorCode.LIMITEXCEEDED).containsTs(ex.getErrorCode())) {
                return new SendStatus.TryAgain(SmsPassportErrorCode.R.toXmlName(ex.getErrorCode()), Option.of(address));

            } else if (Cf.list(
                    SmsPassportErrorCode.PERMANENTBLOCK, SmsPassportErrorCode.PHONEBLOCKED,
                    SmsPassportErrorCode.NOCURRENT, SmsPassportErrorCode.NOROUTE).containsTs(ex.getErrorCode())) {
                return new SendStatus.Failed(SmsPassportErrorCode.R.toXmlName(ex.getErrorCode()), Option.of(address));
            }
            throw ex;
        }
    }

    protected SendStatus doSendEmail(PassportUid uid, Option<Email> email, MailMessage message) {
        String messageId = MailUtils.generateMessageId();

        Option<String> recipient = message.getHeader(HeaderNames.TO);

        if (!recipient.isPresent() && email.isPresent()) {
            recipient = Option.of(new MailAddress(email.get()).getValue());
        }
        if (!recipient.isPresent()) {
            BlackboxCorrectResponse response = blackbox.query().userInfo(
                    LocalhostUtils.localAddress(), uid, Cf.list(), Option.of(EmailsParameterValue.GET_DEFAULT));

            if (response.getDefaultEmail().isPresent()) {
                recipient = Option.of(new MailAddress(response.getDefaultEmail().get()).getValue());
            }
        }
        if (recipient.isPresent()) {
            mailSender.send(message.withDate(Instant.now()).withMessageId(messageId).withTo(recipient.get()));
            return new SendStatus.Sent(messageId, recipient);
        } else {
            return new SendStatus.Failed("no default email", Option.none());
        }
    }

}
