package ru.yandex.reminders.logic.sup;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.util.function.Function;

import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;

import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.IoFunction;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;

public class PushClientSupport<Req, Resp> implements Closeable {

    protected final HttpClient httpClient;
    protected final String url;

    private static final String USER_AGENT = "Yandex.Reminders";

    public PushClientSupport(String url, int maxCons, Timeout timeout) {
        this.url = url;
        this.httpClient = EnvironmentType.PRODUCTION.isActive()
                ? ApacheHttpClientUtils.multiThreadedHttpsClient(timeout, maxCons, USER_AGENT)
                : ApacheHttpClientUtils.trustAllMultiThreadedHttpsClient(timeout, maxCons, maxCons, USER_AGENT);
    }

    protected Resp execute(
            Tuple2List<String, Object> queryParameters, Req body,
            Function<HttpUriRequest, HttpUriRequest> authHandler,
            IoFunction<HttpResponse, Resp> responseParser)
    {
        HttpPost post = new HttpPost(UrlUtils.addParameters(url, queryParameters));

        post.setEntity(new ByteArrayEntity(
                SupClientSettings.benderMapper.serializeJson(body), ContentType.APPLICATION_JSON));

        try {
            return httpClient.execute(authHandler.apply(post), new ResponseParseHandler<>(responseParser));
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    private static class ResponseParseHandler<Resp> implements ResponseHandler<Resp> {

        private final Function<HttpResponse, Resp> parser;

        public ResponseParseHandler(Function<HttpResponse, Resp> parser) {
            this.parser = parser;
        }

        public Resp handleResponse(HttpResponse response) throws IOException {
            int statusCode = response.getStatusLine().getStatusCode();

            if (!ru.yandex.misc.io.http.HttpStatus.is2xx(statusCode)) {
                throw new StrangeResponseException(statusCode, response.getEntity().getContent());
            }
            return parser.apply(response);
        }
    }

    private static class StrangeResponseException extends HttpException {

        public StrangeResponseException(int statusCode, InputStream response) {
            super(statusCode, InputStreamSourceUtils.wrap(response).readText());
        }
    }

    @Override
    public void close() {
        ApacheHttpClientUtils.stopQuietly(httpClient);
    }
}
